// Copyright 2016-2025 Mullvad VPN AB. All Rights Reserved.
// Copyright 2025 Nym Technologies SA <contact@nymtech.net>
// SPDX-License-Identifier: GPL-3.0-only

use std::{io, sync::Arc, time::Duration};

use tokio::sync::{Mutex, watch};
use tokio_util::sync::CancellationToken;

use nym_common::trace_err_chain;
use nym_routing::{CallbackHandle, EventType, RouteManagerHandle, get_best_default_route};
use nym_windows::{
    net::AddressFamily,
    window::{PowerManagementEvent, PowerManagementListener},
};

use super::Connectivity;

#[derive(thiserror::Error, Debug)]
pub enum Error {
    #[error("unable to create listener thread")]
    ThreadCreationError(#[from] io::Error),
    #[error("failed to start connectivity monitor")]
    ConnectivityMonitorError(#[from] nym_routing::Error),
}

pub struct BroadcastListener {
    system_state: Arc<Mutex<SystemState>>,
    _callback_handle: CallbackHandle,
}

unsafe impl Send for BroadcastListener {}

impl BroadcastListener {
    pub async fn start(
        notify_tx: watch::Sender<Connectivity>,
        route_manager: RouteManagerHandle,
        mut power_mgmt_rx: PowerManagementListener,
        shutdown_token: CancellationToken,
    ) -> Result<Self, Error> {
        let (ipv4, ipv6) = Self::check_initial_connectivity();
        let system_state = Arc::new(Mutex::new(SystemState {
            connectivity: ConnectivityInner {
                ipv4,
                ipv6,
                suspended: false,
            },
            notify_tx,
        }));

        let state = system_state.clone();
        tokio::spawn(async move {
            loop {
                tokio::select! {
                    event = power_mgmt_rx.next() => {
                        let Some(event) = event else {
                            break
                        };

                        match event {
                            PowerManagementEvent::Suspend => {
                                tracing::debug!("Machine is preparing to enter sleep mode");
                                state
                                    .lock()
                                    .await
                                    .apply_change(StateChange::Suspended(true));
                            }
                            PowerManagementEvent::ResumeAutomatic => {
                                let state_copy = state.clone();
                                tokio::spawn(async move {
                                    // Tunnel will be unavailable for approximately 2 seconds on a healthy
                                    // machine.
                                    tokio::time::sleep(Duration::from_secs(5)).await;
                                    tracing::debug!(
                                        "Tunnel device is presumed to have been re-initialized"
                                    );
                                    state_copy
                                        .lock()
                                        .await
                                        .apply_change(StateChange::Suspended(false));
                                });
                            }
                            _ => (),
                        }
                    }
                    _ = shutdown_token.cancelled() => {
                        break;
                    }
                }
            }

            tracing::debug!("Offline monitor exiting");
        });

        let callback_handle =
            Self::setup_network_connectivity_listener(system_state.clone(), route_manager).await?;

        Ok(BroadcastListener {
            system_state,
            _callback_handle: callback_handle,
        })
    }

    fn check_initial_connectivity() -> (bool, bool) {
        let v4_connectivity = get_best_default_route(AddressFamily::Ipv4)
            .map(|route| route.is_some())
            .unwrap_or_else(|error| {
                trace_err_chain!(error, "Failed to check initial IPv4 connectivity");
                true
            });
        let v6_connectivity = get_best_default_route(AddressFamily::Ipv6)
            .map(|route| route.is_some())
            .unwrap_or_else(|error| {
                trace_err_chain!(error, "Failed to check initial IPv6 connectivity");
                true
            });

        let is_online = v4_connectivity || v6_connectivity;
        tracing::info!("Initial connectivity: {}", is_offline_str(!is_online));

        (v4_connectivity, v6_connectivity)
    }

    /// The caller must make sure the `system_state` reference is valid
    /// until after `WinNet_DeactivateConnectivityMonitor` has been called.
    async fn setup_network_connectivity_listener(
        system_state: Arc<Mutex<SystemState>>,
        route_manager: RouteManagerHandle,
    ) -> Result<CallbackHandle, Error> {
        let change_handle = route_manager
            .add_default_route_change_callback(Box::new(move |event, addr_family| {
                Self::connectivity_callback(event, addr_family, &system_state)
            }))
            .await
            .map_err(Error::ConnectivityMonitorError)?;
        Ok(change_handle)
    }

    fn connectivity_callback(
        event_type: EventType<'_>,
        family: AddressFamily,
        state_lock: &Arc<Mutex<SystemState>>,
    ) {
        use nym_routing::EventType::*;

        if matches!(event_type, UpdatedDetails(_)) {
            // ignore changes that don't affect the route
            return;
        }

        let connectivity = event_type != Removed;
        let change = match family {
            AddressFamily::Ipv4 => StateChange::NetworkV4Connectivity(connectivity),
            AddressFamily::Ipv6 => StateChange::NetworkV6Connectivity(connectivity),
        };
        let mut state = state_lock.blocking_lock();
        state.apply_change(change);
    }

    #[allow(clippy::unused_async)]
    pub async fn connectivity(&self) -> Connectivity {
        self.system_state
            .lock()
            .await
            .connectivity
            .into_connectivity()
    }
}

#[derive(Debug)]
enum StateChange {
    NetworkV4Connectivity(bool),
    NetworkV6Connectivity(bool),
    Suspended(bool),
}

struct SystemState {
    connectivity: ConnectivityInner,
    notify_tx: watch::Sender<Connectivity>,
}

impl SystemState {
    fn apply_change(&mut self, change: StateChange) {
        let old_state = self.is_offline_currently();
        match change {
            StateChange::NetworkV4Connectivity(connectivity) => {
                self.connectivity.ipv4 = connectivity;
            }
            StateChange::NetworkV6Connectivity(connectivity) => {
                self.connectivity.ipv6 = connectivity;
            }
            StateChange::Suspended(suspended) => {
                self.connectivity.suspended = suspended;
            }
        };

        let new_state = self.connectivity.is_offline();
        if old_state != new_state {
            tracing::info!("Connectivity changed: {}", is_offline_str(new_state));
            if let Err(e) = self.notify_tx.send(self.connectivity.into_connectivity()) {
                tracing::error!("Failed to send new offline state to daemon: {}", e);
            }
        }
    }

    fn is_offline_currently(&self) -> bool {
        self.connectivity.is_offline()
    }
}

// If `offline` is true, return "Offline". Otherwise, return "Connected".
fn is_offline_str(offline: bool) -> &'static str {
    if offline { "Offline" } else { "Connected" }
}

pub type ConnectivityHandle = BroadcastListener;

pub async fn spawn_monitor(
    sender: watch::Sender<Connectivity>,
    route_manager: RouteManagerHandle,
    shutdown_token: CancellationToken,
) -> Result<ConnectivityHandle, Error> {
    let power_mgmt_rx = PowerManagementListener::new();
    BroadcastListener::start(sender, route_manager, power_mgmt_rx, shutdown_token).await
}

#[derive(Clone, Copy, Debug)]
struct ConnectivityInner {
    /// Whether IPv4 connectivity seems to be available on the host.
    ipv4: bool,
    /// Whether IPv6 connectivity seems to be available on the host.
    ipv6: bool,
    /// The host is suspended.
    suspended: bool,
}

impl ConnectivityInner {
    /// Map [`ConnectivityInner`] to the public [`Connectivity`].
    ///
    /// # Note
    ///
    /// If the host is suspended, there is a great likelihood that we should
    /// consider the host to be offline. We synthesize this by setting both
    /// `ipv4` and `ipv6` availability to `false`.
    fn into_connectivity(self) -> Connectivity {
        if self.suspended {
            Connectivity::Status {
                ipv4: false,
                ipv6: false,
            }
        } else {
            Connectivity::Status {
                ipv4: self.ipv4,
                ipv6: self.ipv6,
            }
        }
    }

    /// See [`Connectivity::is_offline`] for details.
    fn is_offline(&self) -> bool {
        self.into_connectivity().is_offline()
    }
}
