package net.nymtech.nymvpn.ui.common.snackbar

import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.SnackbarResult
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.Immutable
import androidx.compose.runtime.MutableState
import androidx.compose.runtime.ReadOnlyComposable
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.runtime.staticCompositionLocalOf
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.platform.LocalContext
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.launch
import net.nymtech.nymvpn.util.StringValue
import kotlin.coroutines.EmptyCoroutineContext

private val LocalSnackbarController = staticCompositionLocalOf {
	SnackbarController(
		host = SnackbarHostState(),
		scope = CoroutineScope(EmptyCoroutineContext),
	)
}
private val channel = Channel<SnackbarChannelMessage?>(capacity = 1)

@Composable
fun SnackbarControllerProvider(content: @Composable (snackbarHost: SnackbarHostState, content: MutableState<CustomSnackbarContent?>) -> Unit) {
	val snackHostState = remember { SnackbarHostState() }
	val scope = rememberCoroutineScope()
	val snackController = remember(scope) { SnackbarController(snackHostState, scope) }
	val context = LocalContext.current
	val customSnackbarData = remember { mutableStateOf<CustomSnackbarContent?>(null) }

	DisposableEffect(snackController, scope) {
		val job = scope.launch {
			for (payload in channel) {
				payload?.let {
					when (it) {
						is SnackbarChannelMessage.Custom -> {
							snackHostState.currentSnackbarData?.dismiss()
							customSnackbarData.value = it.content
							val result = snackHostState.showSnackbar("", duration = customSnackbarData.value?.duration ?: SnackbarDuration.Short)
							if (result == SnackbarResult.ActionPerformed) {
								it.content.action?.onActionPress?.invoke()
							}
							customSnackbarData.value = null
						}

						is SnackbarChannelMessage.Default -> {
							snackController.showMessage(
								message = it.message.asString(context),
								duration = it.duration,
								action = it.action,
							)
						}
					}
				} ?: snackHostState.currentSnackbarData?.dismiss()
			}
		}

		onDispose {
			job.cancel()
		}
	}

	CompositionLocalProvider(LocalSnackbarController provides snackController) {
		content(snackHostState, customSnackbarData)
	}
}

@Immutable
class SnackbarController(
	private val host: SnackbarHostState,
	private val scope: CoroutineScope,
) {
	companion object {
		val current
			@Composable
			@ReadOnlyComposable
			get() = LocalSnackbarController.current

		fun showMessage(message: StringValue, action: SnackbarAction? = null, duration: SnackbarDuration = SnackbarDuration.Short) {
			channel.trySend(
				SnackbarChannelMessage.Default(
					message = message,
					duration = duration,
					action = action,
				),
			)
		}

		fun showMessage(message: String, action: SnackbarAction? = null, duration: SnackbarDuration = SnackbarDuration.Short, iconAction: IconAction?) {
			channel.trySend(
				SnackbarChannelMessage.Custom(
					content = CustomSnackbarContent(
						message = message,
						duration = duration,
						action = action,
						iconAction = iconAction,
					),
				),
			)
		}

		fun dismiss() {
			channel.trySend(null)
		}
	}

	fun showMessage(message: String, action: SnackbarAction? = null, duration: SnackbarDuration = SnackbarDuration.Short) {
		scope.launch {
			/**
			 * note: uncomment this line if you want snackbar to be displayed immediately,
			 * rather than being enqueued and waiting [duration] * current_queue_size
			 */
			host.currentSnackbarData?.dismiss()
			val result = host.showSnackbar(
				message = message,
				actionLabel = action?.title,
				duration = duration,
			)

			if (result == SnackbarResult.ActionPerformed) {
				action?.onActionPress?.invoke()
			}
		}
	}
}

sealed class SnackbarChannelMessage {
	data class Default(
		val message: StringValue,
		val action: SnackbarAction?,
		val duration: SnackbarDuration = SnackbarDuration.Short,
	) : SnackbarChannelMessage()

	data class Custom(
		val content: CustomSnackbarContent,
	) : SnackbarChannelMessage()
}

data class CustomSnackbarContent(
	val message: String,
	val action: SnackbarAction? = null,
	val iconAction: IconAction? = null,
	val duration: SnackbarDuration = SnackbarDuration.Short,
)

data class SnackbarAction(val title: String, val onActionPress: () -> Unit)
data class IconAction(val icon: ImageVector, val onActionPress: () -> Unit)
