#!/bin/bash

# This script updates nym-vpn-app deb package generated by tauri:
# - renames the package name to nym-vpn-app
# - format the version to comply with debian package versioning
# - repacks the package
#
# Deb version spec: https://www.debian.org/doc/debian-policy/ch-controlfields.html#version
#
# The following environment variables have to be set:
# - SRC_DEB: path to the deb package
# - PKGVER: version string (e.g., 1.2.3, 1.2.3-dev.1)

if [ -z "$SRC_DEB" ]; then
    echo "Environment variable SRC_DEB is not set"
    exit 1
fi
if [ -z "$PKGVER" ]; then
    echo "Environment variable PKGVER is not set"
    exit 1
fi

deb_file="$SRC_DEB"
version="$PKGVER"
pkgname="nym-vpn-app"

# Check if the file exists
if [ ! -f "$deb_file" ]; then
    echo "File not found: $deb_file"
    exit 1
fi

echo "version: $version"
# split version string on - | _
IFS='-_' read -r base prerel <<<"$version"
echo "base version: $base"
if [ -n "$prerel" ]; then
  echo "prerelease: $prerel"
  # use ~dev suffix for anything that is not a stable release
  version="${base}~dev"
fi
echo "deb version: $version"

tmpdir=$(mktemp -d)
# Extract the .deb package contents
dpkg-deb -R "$deb_file" "$tmpdir"

# Update the control file
control_file="$tmpdir/DEBIAN/control"
sed -i "s|^Package:.*|Package: $pkgname|" "$control_file"
sed -i "s|^Version:.*|Version: $version|" "$control_file"
cat "$control_file"

# Repack the .deb package with the updated control file
# replace ~ with - in the filename to avoid file renaming by GH
dest="${pkgname}_${version//\~/-}_amd64.deb"
dpkg-deb -Zxz -b "$tmpdir" "$dest"

# Clean up the temporary directory
#rm -rf "$tmpdir"

echo "repacked: $dest"
echo "version: $version"
echo "package=$dest" >> "$GITHUB_OUTPUT"
echo "version=$version" >> "$GITHUB_OUTPUT"
