#!/usr/bin/env bash

set -eu

function usage {
    echo "usage: $0 [clean|build] [--max-concurrent-processes <n>] [solution...]"
    echo "  --max-concurrent-processes <n>  Limit concurrent processes that msbuild can spawn to <n>. Defaults to number of processor cores."
    exit 1
}

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
cd "$SCRIPT_DIR"

source scripts/utils/host
source scripts/utils/log

case $HOST in
    x86_64-pc-windows-msvc) HOST_TARGET=x64;;
    aarch64-pc-windows-msvc) HOST_TARGET=ARM64;;
    *)
        log_error "Unknown Windows target: $HOST"
        exit 1
        ;;
esac

ACTION=build
SOLUTIONS=()

while [[ "$#" -gt 0 ]]; do
    case $1 in
        clean) ACTION="clean";;
        build) ACTION="build";;
        winfw) SOLUTIONS+=(winfw);;
        driverlogic) SOLUTIONS+=(driverlogic);;
        nsis-plugins) SOLUTIONS+=(nsis-plugins);;
        --max-concurrent-processes)
            MAX_CPUS="$2"
            shift
            ;;
        help | --help) usage;;
        *)
            log_error "Unknown parameter: $1"
            exit 1
            ;;
    esac
    shift
done

if [[ -z "${SOLUTIONS[*]}" ]]; then
    SOLUTIONS=(winfw driverlogic nsis-plugins)
fi

BUILD_WINFW=false
BUILD_NSIS=false
BUILD_DRIVERLOGIC=false

for sln in "${SOLUTIONS[@]}"; do
    case $sln in
        winfw) BUILD_WINFW=true;;
        nsis-plugins) BUILD_NSIS=true;;
        driverlogic) BUILD_DRIVERLOGIC=true;;
    esac
done

# List of solution configurations to build.
# Default configurations generated by Visual Studio are "Release" and "Debug".
CPP_BUILD_MODES=${CPP_BUILD_MODES:-"Debug"}
# List of target platforms to build for.
# Common platforms include "x86" and "x64".
CPP_BUILD_TARGETS=${CPP_BUILD_TARGETS:-"$HOST_TARGET"}

function clean_solution {
    local path="$1"
    # Clean all intermediate and output files
    echo "Removing ${path:?}..."
    rm -r "${path:?}/bin/"* 2>/dev/null || true
}

function build_solution_config {
    local sln="$1"
    local config="$2"
    local platform="$3"

    if [ -z ${MAX_CPUS+x} ]; then
        MAX_CPU_COUNT_ARG="/m"
    else
        MAX_CPU_COUNT_ARG="/m:${MAX_CPUS}"
    fi

    set -x
    cmd.exe "/c msbuild.exe $MAX_CPU_COUNT_ARG $(to_win_path "$sln") /p:Configuration=$config /p:Platform=$platform"
    set +x
}

# Builds visual studio solution in all (specified) configurations
function build_solution {
    local path="$1"
    local sln="$1/$2"

    for mode in $CPP_BUILD_MODES; do
        for target in $CPP_BUILD_TARGETS; do
            build_solution_config "$sln" "$mode" "$target"
        done
    done
}

function to_win_path {
    local unixpath="$1"
    # if it's a relative path and starts with a dot (.), don't transform the
    # drive prefix (/c/ -> C:\)
    if echo "$unixpath" | grep '^\.' >/dev/null; then
        echo "$unixpath" | sed -e 's/^\///' -e 's/\//\\/g'
    # if it's an absolute path, transform the drive prefix
    else
        # remove the cygdrive prefix if it's there
        unixpath=$(echo "$unixpath" | sed -e 's/^\/cygdrive//')
        echo "$unixpath" | sed -e 's/^\///' -e 's/\//\\/g' -e 's/^./\0:/'
    fi
}

function get_solution_output_path {
    local solution_root="$1"
    local build_target="$2"
    local build_mode="$3"

    case $build_target in
        "x86") echo "$solution_root/bin/Win32-$build_mode";;
        "x64") echo "$solution_root/bin/x64-$build_mode";;
        "ARM64") echo "$solution_root/bin/ARM64-$build_mode";;
        *)
            echo "Unknown build target: $build_target"
            exit 1
            ;;
    esac
}

function build_nsis_plugins {
    local nsis_root_path="./windows/nsis-plugins"
    build_solution_config "$nsis_root_path/nsis-plugins.sln" "Release" "x86"
}

function clean_all {
    local nsis_root_path="./windows/nsis-plugins"
    clean_solution "$nsis_root_path"

    clean_solution "./windows/libshared"
    clean_solution "./windows/windows-libraries"
    clean_solution "./windows/libwfp"
}

function build {
    if [[ $BUILD_WINFW == "true" ]]; then
        build_solution "./windows/winfw" "winfw.sln"
    fi

    if [[ $BUILD_DRIVERLOGIC == "true" ]]; then
        build_solution "./windows/driverlogic" "driverlogic.sln"
    fi

    if [[ $BUILD_NSIS == "true" ]]; then
        build_nsis_plugins
    fi
}

case $ACTION in
    "build") build;;
    "clean") clean_all;;
    *)
        echo "Unknown build action: $ACTION"
        exit 1
        ;;
esac
