/*
 * Copyright (c) 2025 Alwin Ibba
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.ibbaa.keepitup.ui.dialog;

import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.action.ViewActions.click;
import static androidx.test.espresso.action.ViewActions.longClick;
import static androidx.test.espresso.action.ViewActions.replaceText;
import static androidx.test.espresso.action.ViewActions.swipeUp;
import static androidx.test.espresso.assertion.ViewAssertions.matches;
import static androidx.test.espresso.matcher.ViewMatchers.isDisplayed;
import static androidx.test.espresso.matcher.ViewMatchers.isRoot;
import static androidx.test.espresso.matcher.ViewMatchers.withId;
import static androidx.test.espresso.matcher.ViewMatchers.withText;
import static junit.framework.TestCase.assertTrue;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;

import android.os.Bundle;

import androidx.test.core.app.ActivityScenario;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import androidx.test.filters.MediumTest;

import net.ibbaa.keepitup.R;
import net.ibbaa.keepitup.model.FileEntry;
import net.ibbaa.keepitup.test.mock.MockClipboardManager;
import net.ibbaa.keepitup.ui.BaseUITest;
import net.ibbaa.keepitup.ui.GlobalSettingsActivity;
import net.ibbaa.keepitup.ui.adapter.FileEntryAdapter;
import net.ibbaa.keepitup.util.BundleUtil;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

import java.io.File;
import java.io.IOException;

@MediumTest
@RunWith(AndroidJUnit4.class)
public class FileChooseDialogFileModeTest extends BaseUITest {

    private ActivityScenario<?> activityScenario;
    private String root;

    @Before
    public void beforeEachTestMethod() {
        super.beforeEachTestMethod();
        root = getFileManager().getExternalRootDirectory(0).getAbsolutePath();
        createTestFiles();
        activityScenario = launchSettingsInputActivity(GlobalSettingsActivity.class, getBypassSystemSAFBundle());
        deleteLogFolder();
    }

    @After
    public void afterEachTestMethod() {
        super.afterEachTestMethod();
        activityScenario.close();
    }

    private void deleteLogFolder() {
        getFileManager().delete(getFileManager().getExternalDirectory("log", 0));
    }

    @Test
    public void testDisplayInitialFileListLevel1FileExists() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1FileExistsScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1FileDoesNotExist() {
        FileChooseDialog dialog = openFileChooseDialog("", "file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1FileDoesNotExistScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1FileIsFolder() {
        FileChooseDialog dialog = openFileChooseDialog("", "folder1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1")));
        assertEquals("", dialog.getFolder());
        assertEquals("folder1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1FileIsFolderExistScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "folder2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2")));
        assertEquals("", dialog.getFolder());
        assertEquals("folder2", dialog.getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2")));
        assertEquals("", getDialog().getFolder());
        assertEquals("folder2", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2")));
        assertEquals("", getDialog().getFolder());
        assertEquals("folder2", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1Empty() {
        getFileManager().delete(getFileManager().getExternalRootDirectory(0));
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(1)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertFalse(adapter.isItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel1EmptyScreenRotation() {
        getFileManager().delete(getFileManager().getExternalRootDirectory(0));
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(1)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertFalse(adapter.isItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(1)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        adapter = getAdapter();
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertFalse(adapter.isItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(1)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        adapter = getAdapter();
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertFalse(adapter.isItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel2FileExists() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel2FileExistsScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel2FileDoesNotExist() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file2")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file2", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel2FileDoesNotExistScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file2")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file2", dialog.getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file2", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file2")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file2", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel2FileIsFolder() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_folder1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_folder1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_folder1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testDisplayInitialFileListLevel2FileIsFolderScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_folder2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_folder2")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_folder2", dialog.getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_folder2", dialog.getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_folder2")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_folder2", getDialog().getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel1InitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel1InitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel1NoInitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "XYZ");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/XYZ")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("XYZ")));
        assertEquals("", dialog.getFolder());
        assertEquals("XYZ", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel1NoInitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel2InitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "folder2_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file1", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel2InitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "folder2_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel2NoInitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileLevel2NoInitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel1InitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel1InitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel1NoInitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel1NoInitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel2InitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "folder2_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_folder1/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2/folder2_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2/folder2_folder1", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel2InitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "folder2_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_folder1/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2/folder2_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2/folder2_folder1", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_folder1/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2/folder2_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2/folder2_folder1", getDialog().getFolder());
        assertEquals("folder2_file2", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel2NoInitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2/folder2_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2/folder2_folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file1", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectFileAndFolderLevel2NoInitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2/folder2_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2/folder2_folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2/folder2_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder2/folder2_folder1", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("folder2_file1", getDialog().getFile());
        adapter = getAdapter();
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel1InitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1/folder1_folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel1InitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1/folder1_folder1", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel1NoInitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel1NoInitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel2InitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1/folder1_folder2", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel2InitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1/folder1_folder2", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1/folder1_folder2", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("folder1_file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel2NoInitialSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1/folder1_folder2", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testOpenFolderLevel2NoInitialSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "xyz");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1/folder1_folder2", dialog.getFolder());
        assertEquals("xyz", dialog.getFile());
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1/folder1_folder2", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("xyz")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("xyz", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel1InputAndSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file2", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel1InputAndSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel1InputAndSelect() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("xyz"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1", true, false, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file2")));
        assertEquals("", dialog.getFolder());
        assertEquals("file2", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file2", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel1InputAndSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("xyz"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(4);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder2", true, false, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel1InputAndOpen() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel1InputAndOpenScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel1InputAndOpen() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("xyz"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(4);
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("folder2"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder2_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder2_file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder2_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel1InputAndOpenScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("xyz"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(4);
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("folder2"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel1CopyPaste() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        assertEquals("file3", clipboardManager.getData());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel1CopyPasteScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        rotateScreen(activityScenario);
        deleteLogFolder();
        clipboardManager = prepareMockClipboardManager(getDialog());
        clipboardManager.putData("file3");
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel1CopyPaste() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        assertEquals("file3", clipboardManager.getData());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(1)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("file3")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("file3", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        assertEquals("file3", clipboardManager.getData());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel1CopyPasteScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        rotateScreen(activityScenario);
        deleteLogFolder();
        clipboardManager = prepareMockClipboardManager(getDialog());
        clipboardManager.putData("file3");
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(1)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("file3")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("file3", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        assertEquals("file3", clipboardManager.getData());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel2InputAndSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1/folder1_folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel2InputAndSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("..", true, true, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("..", true, true, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel2InputAndSelect() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("folderxy"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folderxy/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folderxy")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folderxy", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(3);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder2/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1/folder1_folder2", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel2InputAndSelectScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("folderxy"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folderxy/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folderxy")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folderxy", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        selectEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("..", true, true, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("..", true, true, true)));
        assertTrue(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertTrue(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel2InputAndOpen() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(2);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1/folder1_folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1/folder1_folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(1)));
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel2InputAndOpenScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "folder1_file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/folder1_file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder1_file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("folder1_file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel2InputAndOpen() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "folder2_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("xyz"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder2_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder2_file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder2_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText(""));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(5)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withText("folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 4))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel2InputAndOpenScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "folder2_file2");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/folder2_file2")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("folder2_file2")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("folder2_file2", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file25"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file25", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText("xyz"));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/xyz/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("xyz")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("xyz", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(replaceText(""));
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file25")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file25")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file25", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(5, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(4), getFileEntry("folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel2CopyPaste() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        assertEquals("file3", clipboardManager.getData());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder2_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder2_file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder2_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileLevel2CopyPasteScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder1", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder1", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        rotateScreen(activityScenario);
        deleteLogFolder();
        clipboardManager = prepareMockClipboardManager(getDialog());
        clipboardManager.putData("file3");
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder1/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder1")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("folder1", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder1_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder1_folder1", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder1_folder2", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder1_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder1_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder1_folder2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel2CopyPaste() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        assertEquals("file3", clipboardManager.getData());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("file3")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("file3", dialog.getFolder());
        assertEquals("file3", dialog.getFile());
        assertEquals("file3", clipboardManager.getData());
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder2_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder2_file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder2_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testFileAndFolderLevel2CopyPasteScreenRotation() {
        FileChooseDialog dialog = openFileChooseDialog("folder2", "file1");
        MockClipboardManager clipboardManager = prepareMockClipboardManager(dialog);
        clipboardManager.putData("file3");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder2", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(longClick());
        rotateScreen(activityScenario);
        deleteLogFolder();
        clipboardManager = prepareMockClipboardManager(getDialog());
        clipboardManager.putData("file3");
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.listview_dialog_context_options)).check(matches(withListSize(2)));
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        onView(withId(R.id.edittext_dialog_file_choose_folder)).perform(longClick());
        assertEquals(2, getActivity(activityScenario).getSupportFragmentManager().getFragments().size());
        onView(withId(R.id.textview_dialog_context_options_title)).check(matches(withText("Text options")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 0))).check(matches(withText("Copy")));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).check(matches(withText("Paste")));
        onView(withId(R.id.imageview_dialog_context_options_cancel)).check(matches(isDisplayed()));
        onView(allOf(withId(R.id.textview_list_item_context_option_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_context_options), 1))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file3/file3")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("file3")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file3")));
        assertEquals("file3", getDialog().getFolder());
        assertEquals("file3", getDialog().getFile());
        assertEquals("file3", clipboardManager.getData());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(4, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("folder2_file1", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("folder2_file2", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getItem(3), getFileEntry("folder2_folder1", true, false, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.listview_dialog_file_choose_file_entries)).check(matches(withListSize(4)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withText("..")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 0))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(withText("folder2_file1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 1))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withDrawable(R.drawable.icon_file)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(withText("folder2_file2")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 2))).check(matches(not(isDisplayed())));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_symbol), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder)));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withText("folder2_folder1")));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), 3))).check(matches(withDrawable(R.drawable.icon_folder_open_shadow)));
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testIdenticalFolderNameOpen() {
        getFileManager().delete(getFileManager().getExternalRootDirectory(0));
        getFileManager().delete(getFileManager().getExternalRootDirectory(1));
        createIdenticalNameTestFiles();
        FileChooseDialog dialog = openFileChooseDialog("", "file");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        FileEntryAdapter adapter = dialog.getAdapter();
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/download/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("download")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("download", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/download/download/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("download/download")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("download/download", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(1);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/download/download/download/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("download/download/download")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("download/download/download", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        assertEquals(1, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertFalse(adapter.isItemSelected());
        assertFalse(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/download/download/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("download/download")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("download/download", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/download/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("download")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("download", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("", dialog.getFolder());
        assertEquals("file", dialog.getFile());
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testIdenticalFolderNameOpenScreenRotation() {
        getFileManager().delete(getFileManager().getExternalRootDirectory(0));
        getFileManager().delete(getFileManager().getExternalRootDirectory(1));
        createIdenticalNameTestFiles();
        openFileChooseDialog("", "file");
        openEntry(1);
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/download/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("download")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("download", getDialog().getFolder());
        assertEquals("file", getDialog().getFile());
        FileEntryAdapter adapter = getAdapter();
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, true)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        openEntry(0);
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file", getDialog().getFile());
        adapter = getAdapter();
        assertEquals(3, adapter.getItemCount());
        assertTrue(areEntriesEqual(adapter.getItem(0), getFileEntry("..", true, true, false)));
        assertTrue(areEntriesEqual(adapter.getItem(1), getFileEntry("download", true, false, true)));
        assertTrue(areEntriesEqual(adapter.getItem(2), getFileEntry("file", false, false, false)));
        assertTrue(areEntriesEqual(adapter.getSelectedItem(), getFileEntry("file", false, false, false)));
        assertTrue(adapter.isItemSelected());
        assertTrue(adapter.isFileItemSelected());
        assertFalse(adapter.isFolderItemSelected());
        onView(withId(R.id.imageview_dialog_file_choose_cancel)).perform(click());
    }

    @Test
    public void testSelectSwipeScreenRotation() {
        openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.scrollview_dialog_file_choose)).perform(swipeUp());
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withText("folder2"))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.scrollview_dialog_file_choose)).perform(swipeUp());
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withText("folder2"))).perform(click());
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/folder2/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("folder2")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("folder2", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
    }

    @Test
    public void testEmptyFilename() {
        openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText(""));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        onView(allOf(withText("Filename"), withGridLayoutPosition(1, 0))).check(matches(isDisplayed()));
        onView(allOf(withText("No value specified"), withGridLayoutPosition(1, 1))).check(matches(isDisplayed()));
        onView(withId(R.id.imageview_dialog_validator_error_ok)).perform(click());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file1"));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        assertTrue(getActivity(activityScenario).getSupportFragmentManager().getFragments().isEmpty());
    }

    @Test
    public void testEmptyFilenameScreenRotation() {
        openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText(""));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(allOf(withText("Filename"), withGridLayoutPosition(1, 0))).check(matches(isDisplayed()));
        onView(allOf(withText("No value specified"), withGridLayoutPosition(1, 1))).check(matches(isDisplayed()));
        onView(withId(R.id.imageview_dialog_validator_error_ok)).perform(click());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file1"));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        assertTrue(getActivity(activityScenario).getSupportFragmentManager().getFragments().isEmpty());
    }

    @Test
    public void testEmptyFilenameAllowed() {
        FileChooseDialog dialog = openFileChooseDialog("", "file1", FileChooseDialog.Mode.FILE_ALLOW_EMPTY);
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", dialog.getFolder());
        assertEquals("file1", dialog.getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText(""));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        assertTrue(getActivity(activityScenario).getSupportFragmentManager().getFragments().isEmpty());
        assertEquals("", dialog.getFile());
    }

    @Test
    public void testInvalidFilename() {
        openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("test/"));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withTextColor(R.color.textErrorColor)));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        onView(allOf(withText("Filename"), withGridLayoutPosition(1, 0))).check(matches(isDisplayed()));
        onView(allOf(withText("No valid filename"), withGridLayoutPosition(1, 1))).check(matches(isDisplayed()));
        onView(withId(R.id.imageview_dialog_validator_error_ok)).perform(click());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file1"));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withTextColor(R.color.textColor)));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        assertTrue(getActivity(activityScenario).getSupportFragmentManager().getFragments().isEmpty());
    }

    @Test
    public void testInvalidFilenameScreenRotation() {
        openFileChooseDialog("", "file1");
        onView(withId(R.id.textview_dialog_file_choose_absolute)).check(matches(withText(root + "/file1")));
        onView(withId(R.id.edittext_dialog_file_choose_folder)).check(matches(withText("")));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withText("file1")));
        assertEquals("", getDialog().getFolder());
        assertEquals("file1", getDialog().getFile());
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("test/"));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withTextColor(R.color.textErrorColor)));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(allOf(withText("Filename"), withGridLayoutPosition(1, 0))).check(matches(isDisplayed()));
        onView(allOf(withText("No valid filename"), withGridLayoutPosition(1, 1))).check(matches(isDisplayed()));
        onView(withId(R.id.imageview_dialog_validator_error_ok)).perform(click());
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withTextColor(R.color.textErrorColor)));
        rotateScreen(activityScenario);
        deleteLogFolder();
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withTextColor(R.color.textErrorColor)));
        onView(withId(R.id.edittext_dialog_file_choose_file)).perform(replaceText("file1"));
        onView(withId(R.id.edittext_dialog_file_choose_file)).check(matches(withTextColor(R.color.textColor)));
        onView(withId(R.id.imageview_dialog_file_choose_ok)).perform(click());
        assertTrue(getActivity(activityScenario).getSupportFragmentManager().getFragments().isEmpty());
    }

    private FileChooseDialog openFileChooseDialog(String folder, String file) {
        return openFileChooseDialog(folder, file, FileChooseDialog.Mode.FILE);
    }

    private FileChooseDialog openFileChooseDialog(String folder, String file, FileChooseDialog.Mode mode) {
        FileChooseDialog fileChooseDialog = new FileChooseDialog();
        Bundle bundle = BundleUtil.stringsToBundle(new String[]{fileChooseDialog.getFolderRootKey(), fileChooseDialog.getFolderKey()}, new String[]{root, folder});
        bundle = BundleUtil.stringToBundle(fileChooseDialog.getFileModeKey(), mode.name(), bundle);
        bundle = BundleUtil.stringToBundle(fileChooseDialog.getFileKey(), file, bundle);
        fileChooseDialog.setArguments(bundle);
        fileChooseDialog.show(getActivity(activityScenario).getSupportFragmentManager(), GlobalSettingsActivity.class.getName());
        onView(isRoot()).perform(waitFor(1000));
        return fileChooseDialog;
    }

    private void openEntry(int index) {
        onView(isRoot()).perform(waitFor(1000));
        onView(allOf(withId(R.id.imageview_list_item_file_entry_open), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), index))).perform(click());
    }

    private void selectEntry(int index) {
        onView(isRoot()).perform(waitFor(1000));
        onView(allOf(withId(R.id.textview_list_item_file_entry_name), withChildDescendantAtPosition(withId(R.id.listview_dialog_file_choose_file_entries), index))).perform(click());
    }

    private FileChooseDialog getDialog() {
        return (FileChooseDialog) getDialog(activityScenario, FileChooseDialog.class);
    }

    private FileEntryAdapter getAdapter() {
        return getDialog().getAdapter();
    }

    private void createIdenticalNameTestFiles() {
        try {
            File downloadFolder1 = new File(root, "download");
            File file1 = new File(root, "file");
            assertTrue(downloadFolder1.mkdir());
            assertTrue(file1.createNewFile());
            File downloadFolder2 = new File(downloadFolder1, "download");
            File File2 = new File(downloadFolder1, "file");
            assertTrue(downloadFolder2.mkdirs());
            assertTrue(File2.createNewFile());
            File downloadFolder3 = new File(downloadFolder2, "download");
            File File3 = new File(downloadFolder2, "file");
            assertTrue(downloadFolder3.mkdirs());
            assertTrue(File3.createNewFile());
        } catch (IOException exc) {
            throw new RuntimeException(exc);
        }
    }

    private void createTestFiles() {
        try {
            File folder1 = new File(root, "folder1");
            File file1 = new File(root, "file1");
            File file2 = new File(root, "file2");
            File folder2 = new File(root, "folder2");
            assertTrue(folder1.mkdir());
            assertTrue(file1.createNewFile());
            assertTrue(file2.createNewFile());
            assertTrue(folder2.mkdir());
            File folder1Folder1 = new File(folder1, "folder1_folder1");
            File folder1Folder2 = new File(folder1, "folder1_folder2");
            File folder1File1 = new File(folder1, "folder1_file1");
            assertTrue(folder1Folder1.mkdir());
            assertTrue(folder1Folder2.mkdir());
            assertTrue(folder1File1.createNewFile());
            File folder2Folder1 = new File(folder2, "folder2_folder1");
            File folder2File1 = new File(folder2, "folder2_file1");
            File folder2File2 = new File(folder2, "folder2_file2");
            assertTrue(folder2Folder1.mkdir());
            assertTrue(folder2File1.createNewFile());
            assertTrue(folder2File2.createNewFile());
        } catch (IOException exc) {
            throw new RuntimeException(exc);
        }
    }

    private FileEntry getFileEntry(String name, boolean directory, boolean parent, boolean canVisit) {
        FileEntry fileEntry = new FileEntry();
        fileEntry.setName(name);
        fileEntry.setDirectory(directory);
        fileEntry.setParent(parent);
        fileEntry.setCanVisit(canVisit);
        return fileEntry;
    }

    private boolean areEntriesEqual(FileEntry entry1, FileEntry entry2) {
        if (!entry1.getName().equals(entry2.getName())) {
            return false;
        }
        if (entry1.isDirectory() != entry2.isDirectory()) {
            return false;
        }
        if (entry1.canVisit() != entry2.canVisit()) {
            return false;
        }
        return entry1.isParent() == entry2.isParent();
    }

    private MockClipboardManager prepareMockClipboardManager(FileChooseDialog folderChooseDialog) {
        MockClipboardManager clipboardManager = new MockClipboardManager();
        clipboardManager.clearData();
        folderChooseDialog.injectClipboardManager(clipboardManager);
        return clipboardManager;
    }
}
