#!/usr/bin/env python
# SPDX-License-Identifier: GPL-3.0-or-later
# Parses emoji-sequences.txt and emoji-zwj-sequences.txt and turns it into C++.
# You should be able to get those here: https://unicode.org/Public/emoji/latest/
# Formatted with black.
import fileinput
import re

non_blank_re = re.compile(r"\S")
comment_re = re.compile(r"^\s*#")
emoji_sequence_re = re.compile(r"^((?:\s*[0-9a-fA-F]+(?:\.\.[0-9a-fA-F]+)?)+)\s*;")
whitespace_re = re.compile(r"\s+")
codepoint_re = re.compile(r"^([0-9a-fA-F]+)(?:\.\.([0-9a-fA-F]+))?$")


class MatcherGenerator:
    def __init__(self, states):
        self._states = states
        self._ids = []
        self._next_id = 0

    def _get_id(self, states):
        for i, s in self._ids:
            if s is states:
                return i
        next_id = self._next_id
        self._next_id = next_id + 1
        self._ids.append((next_id, states))
        return next_id

    def _generate_recursive(self, states, pos, count):
        is_last = pos == count - 1
        print(f"\t\tcase {self._get_id(states)}:")
        print(f"\t\t\tswitch(input[offset + {pos}].unicode()) {{")

        for char in sorted(states.keys()):
            next_states = states[char]
            print(f"\t\t\tcase 0x{char:04x}u:")
            if not is_last:
                next_id = self._get_id(next_states)
                print(f"\t\t\t\tstate = {next_id};")
                print("\t\t\t\tcontinue;")

        if is_last:
            print("\t\t\t\treturn true;")

        print(f"\t\t\tdefault:")
        print("\t\t\t\treturn false;")
        print("\t\t\t}")

        for char in sorted(states.keys()):
            next_states = states[char]
            if next_states:
                assert not is_last
                self._generate_recursive(next_states, pos + 1, count)
            else:
                assert is_last

    def generate(self, count):
        print(f"static bool emojiMatch{count}(const QString &input, int offset)")
        print("{")
        print(f"\tint state = 0;")
        print("\twhile(true) {")
        print("\t\tswitch(state) {")
        self._generate_recursive(self._states, 0, count)
        print("\t\tdefault:")
        print(f'\t\t\tQ_ASSERT_X(false, "emojiMatch{count}", "Invalid state");')
        print("\t\t\treturn false;")
        print("\t\t}")
        print("\t}")
        print("}")
        print()


class Context:
    def __init__(self):
        self._by_count = {}

    @property
    def _max_count(self):
        return max(self._by_count.keys())

    def _add_recursive(self, states, chars):
        if chars:
            self._add_recursive(states.setdefault(chars[0], {}), chars[1:])

    def add(self, chars):
        count = len(chars)
        states = self._by_count.setdefault(count, {})
        self._add_recursive(states, chars)

    def generate(self):
        print("// SPDX-License-Identifier: GPL-3.0-or-later")
        print("// This file is generated by parse-emoji.py, don't edit it manually!")
        print("// It is based on emoji-sequences.txt and emoji-zwj-sequences.txt.")
        print("#include <QString>")
        print()
        print("namespace htmlutils {")
        print()

        for count in sorted(self._by_count.keys()):
            MatcherGenerator(self._by_count[count]).generate(count)

        print("QString emojify(const QString &in, const QString &replacement)")
        print("{")
        print("\tQString out;")
        print("\tint len = in.length();")
        print("\tint pos = 0;")
        print("\tint i = 0;")
        print("\twhile(i < len) {")
        print("\t\tint remaining = len - i;")
        print("\t\t", end="")
        for count in sorted(self._by_count.keys(), reverse=True):
            print(f"if(remaining >= {count} && emojiMatch{count}(in, i)) {{")
            print("\t\t\tout.append(in.mid(pos, i - pos));")
            print(f"\t\t\tout.append(replacement.arg(in.mid(i, {count})));")
            print(f"\t\t\ti += {count};")
            print("\t\t\tpos = i;")
            print("\t\t} else ", end="")
        print("{")
        print("\t\t\t++i;")
        print("\t\t}")
        print("\t}")
        print()
        print("\tif(pos == 0) {")
        print("\t\treturn in;")
        print("\t} else {")
        print("\t\tout.append(in.mid(pos));")
        print("\t\treturn out;")
        print("\t}")
        print("}")
        print()
        print("}")


def should_parse(line):
    return non_blank_re.search(line) and not comment_re.search(line)


def add_codepoints(c, chars, codepoints):
    if codepoints:
        m = codepoint_re.match(codepoints[0])
        first = int(m.group(1), 16)
        last = int(m.group(2), 16) if m.group(2) else first
        for codepoint in range(first, last + 1):
            encoded = chr(codepoint).encode("utf_16_le")
            i = 0
            next_chars = chars[:]
            while i < len(encoded):
                next_chars.append(
                    int.from_bytes(encoded[i : i + 2], "little", signed=False)
                )
                i += 2
            add_codepoints(c, next_chars, codepoints[1:])
    else:
        c.add(chars)


def parse_emoji_sequence(c, line):
    m = emoji_sequence_re.search(line)
    if not m:
        raise ValueError(f"Can't parse line '{line.rstrip()}")
    codepoints = whitespace_re.split(m.group(1).strip())
    add_codepoints(c, [], codepoints)


c = Context()
for line in fileinput.input(
    files=("emoji-sequences.txt", "emoji-zwj-sequences.txt"), encoding="utf-8"
):
    if should_parse(line):
        parse_emoji_sequence(c, line)
c.generate()
