// SPDX-License-Identifier: GPL-3.0-or-later
#ifndef THINSRV_MULTISERVER_H
#define THINSRV_MULTISERVER_H
#include "libserver/jsonapi.h"
#include "libserver/sslserver.h"
#include <QDateTime>
#include <QHostAddress>
#include <QObject>

class QDir;
class QTcpServer;
class QWebSocketServer;

namespace server {

class Client;
class ExtBans;
class Session;
class SessionServer;
class ServerConfig;
class ThinServerClient;

/**
 * The drawpile server.
 */
class MultiServer final : public QObject {
	Q_OBJECT
public:
	// Takes ownership of config.
	explicit MultiServer(ServerConfig *config, QObject *parent = nullptr);

	void setSslCertFile(
		const QString &certfile, const QString &keyfile,
		SslServer::Algorithm keyAlgorithm)
	{
		m_sslCertFile = certfile;
		m_sslKeyFile = keyfile;
		m_sslKeyAlgorithm = keyAlgorithm;
	}

	void setAutoStop(bool autostop);
	void setRecordingPath(const QString &path);
	void setTemplateDirectory(const QDir &dir);

	/**
	 * @brief Get the port the server is running from
	 * @return port number or zero if server is not running
	 */
	int port() const { return m_port; }

	Q_INVOKABLE bool isRunning() const { return m_state != STOPPED; }

	SessionServer *sessionServer() { return m_sessions; }

	ServerConfig *config() { return m_config; }

public slots:
	void setSessionDirectory(const QDir &dir);

	//! Start the server on the given port and listening address
	bool start(
		quint16 tcpPort, const QHostAddress &tcpAddress = QHostAddress::Any,
		quint16 webSocketPort = 0,
		const QHostAddress &webSocketAddress = QHostAddress::Any);

	//! Stop the server. All clients are disconnected.
	void stop();

	/**
	 * @brief Call the server's JSON administration API
	 *
	 * This is used by the HTTP admin API.
	 *
	 * @param method query method
	 * @param path path components
	 * @param request request body content
	 * @return JSON API response content
	 */
	JsonApiResult callJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request);

	/**
	 * @brief Like callJsonApi(), but the jsonApiResponse signal will be emitted
	 * instead of the return value
	 */
	void callJsonApiAsync(
		const QString &requestId, JsonApiMethod method, const QStringList &path,
		const QJsonObject &request);

private slots:
	void newTcpClient();
	void newWebSocketClient();
	void printStatusUpdate();
	void tryAutoStop();

	/**
	 * @brief Assign a recording file name to a new session
	 *
	 * The name is generated by replacing placeholders in the file name pattern.
	 * If a file with the same name exists, a number is inserted just before the
	 * suffix.
	 *
	 * If the file name pattern points to a directory, the default pattern "%d
	 * %t session %i.dprec" will be used.
	 *
	 * The following placeholders are supported:
	 *
	 *  ~/ - user's home directory (at the start of the pattern)
	 *  %d - the current date (YYYY-MM-DD)
	 *  %h - the current time (HH.MM.SS)
	 *  %i - session ID
	 *  %a - session alias (or ID if not assigned)
	 *
	 * @param session
	 */
	void assignRecording(Session *session);

signals:
	void serverStartError(const QString &message);
	void serverStarted();
	void serverStopped();
	void jsonApiResult(const QString &serverId, const JsonApiResult &result);
	void userCountChanged(int count);

private:
	enum State { RUNNING, STOPPING, STOPPED };

	bool createServer();
	bool createServer(bool enableWebSockets);
	bool abortStart();
	void updateInternalConfig();

	void newClient(ThinServerClient *client);

	JsonApiResult callJsonApiCheckLock(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, const QString &section,
		const std::function<JsonApiResult(
			MultiServer *, JsonApiMethod, const QStringList &,
			const QJsonObject &, bool)> &fn,
		const QSet<QStringList> &allowedPaths = {});
	JsonApiResult serverJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult statusJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request);
	JsonApiResult sessionsJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult usersJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult banlistJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult systembansJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult userbansJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult listserverWhitelistJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult accountsJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult logJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult extbansJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request, bool sectionLocked);
	JsonApiResult locksJsonApi(
		JsonApiMethod method, const QStringList &path,
		const QJsonObject &request);

	ServerConfig *m_config;
	QTcpServer *m_tcpServer;
	QWebSocketServer *m_webSocketServer;
	SessionServer *m_sessions;
	ExtBans *m_extBans;
	State m_state;
	bool m_autoStop;
	int m_port;
	QString m_sslCertFile;
	QString m_sslKeyFile;
	SslServer::Algorithm m_sslKeyAlgorithm;
	QString m_recordingPath;
	QDateTime m_started;
};

}

#endif
