/*
 * Copyright (C) 2022 askmeaboutloom
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * --------------------------------------------------------------------
 *
 * This code is based on Drawpile, using it under the GNU General Public
 * License, version 3. See 3rdparty/licenses/drawpile/COPYING for details.
 */
#ifndef DPENGINE_TILE_H
#define DPENGINE_TILE_H
#include "pixels.h"
#include <dpcommon/common.h>
#include <dpcommon/memory_pool.h>

typedef struct DP_DrawContext DP_DrawContext;
typedef struct DP_Image DP_Image;
typedef struct ZSTD_CCtx_s ZSTD_CCtx;
typedef struct ZSTD_DCtx_s ZSTD_DCtx;

#define DP_TILE_BYTES            (DP_TILE_LENGTH * sizeof(DP_Pixel15))
#define DP_TILE_COMPRESSED_BYTES (DP_TILE_LENGTH * sizeof(DP_Pixel8))

typedef struct DP_TileCounts {
    int x, y;
} DP_TileCounts;

#ifdef DP_NO_STRICT_ALIASING

typedef struct DP_Tile DP_Tile;
typedef struct DP_TransientTile DP_TransientTile;

#else

typedef struct DP_Tile DP_Tile;
typedef struct DP_Tile DP_TransientTile;

#endif


DP_INLINE int DP_tile_size_round_up(int i)
{
    return (i + DP_TILE_SIZE - 1) / DP_TILE_SIZE;
}

DP_INLINE int DP_tile_size_round_down(int i)
{
    return (i / DP_TILE_SIZE) * DP_TILE_SIZE;
}

DP_INLINE int DP_tile_count_round(int i)
{
    return i < 0 ? (i - DP_TILE_SIZE + 1) / DP_TILE_SIZE
                 : (i + DP_TILE_SIZE - 1) / DP_TILE_SIZE;
}

DP_INLINE DP_TileCounts DP_tile_counts_round(int width, int height)
{
    DP_TileCounts tc = {DP_tile_count_round(width),
                        DP_tile_count_round(height)};
    return tc;
}

DP_INLINE int DP_tile_total_round(int width, int height)
{
    DP_TileCounts tile_counts = DP_tile_counts_round(width, height);
    return tile_counts.x * tile_counts.y;
}


const uint16_t *DP_tile_opaque_mask(void);


DP_MemoryPoolStatistics DP_tile_memory_usage(void);


DP_Tile *DP_tile_new(unsigned int context_id);

DP_Tile *DP_tile_new_from_pixel15(unsigned int context_id, DP_Pixel15 pixel);

DP_Tile *DP_tile_new_from_upixel15(unsigned int context_id, DP_UPixel15 pixel);

DP_Tile *DP_tile_new_from_pixels8(unsigned int context_id,
                                  const DP_Pixel8 *pixels);

DP_Tile *DP_tile_new_from_bgra(unsigned int context_id, uint32_t bgra);

DP_Tile *DP_tile_new_from_deflate(DP_DrawContext *dc, unsigned int context_id,
                                  const unsigned char *image,
                                  size_t image_size);

DP_Tile *DP_tile_new_from_split_delta_zstd8le_with(
    ZSTD_DCtx **in_out_ctx_or_null, DP_SplitTile8 *split_tile8_buffer,
    unsigned int context_id, const unsigned char *image, size_t image_size);

DP_Tile *DP_tile_new_from_split_delta_zstd8le(DP_DrawContext *dc,
                                              unsigned int context_id,
                                              const unsigned char *image,
                                              size_t image_size);

DP_Tile *DP_tile_new_mask_from_delta_zstd8le(DP_DrawContext *dc,
                                             unsigned int context_id,
                                             const unsigned char *mask,
                                             size_t mask_size);

DP_Tile *DP_tile_new_zebra(unsigned int context_id, DP_Pixel15 pixel1,
                           DP_Pixel15 pixel2);

DP_Tile *DP_tile_censored_noinc(void);

DP_Tile *DP_tile_censored_inc(void);

DP_Tile *DP_tile_opaque_noinc(void);

DP_Tile *DP_tile_opaque_inc(void);

// Is the given tile identical to the tile returned by DP_tile_opaque_(no)inc?
bool DP_tile_opaque_ident(DP_Tile *tile);

DP_Tile *DP_tile_incref(DP_Tile *tile);

DP_Tile *DP_tile_incref_nullable(DP_Tile *tile_or_null);

DP_Tile *DP_tile_incref_by(DP_Tile *tile, int refcount);

DP_Tile *DP_tile_incref_by_nullable(DP_Tile *tile_or_null, int refcount);

void DP_tile_decref(DP_Tile *tile);

void DP_tile_decref_nullable(DP_Tile *tile_or_null);

int DP_tile_refcount(DP_Tile *tile);

bool DP_tile_transient(DP_Tile *tile);


unsigned int DP_tile_context_id(DP_Tile *tile);

const DP_Pixel15 *DP_tile_pixels(DP_Tile *tile);

DP_Pixel15 DP_tile_pixel_at(DP_Tile *tile, int x, int y);

bool DP_tile_blank(DP_Tile *tile);

bool DP_tile_opaque(DP_Tile *tile_or_null);

bool DP_tile_same_pixel(DP_Tile *tile_or_null, DP_Pixel15 *out_pixel);

bool DP_tile_pixels_equal(DP_Tile *t1, DP_Tile *t2);

bool DP_tile_pixels_equal_pixel(DP_Tile *tile, DP_Pixel15 pixel);


size_t DP_tile_compress_pixel8be(DP_Pixel15 pixel,
                                 unsigned char *(*get_output_buffer)(size_t,
                                                                     void *),
                                 void *user);

size_t DP_tile_compress_pixel8le(DP_Pixel15 pixel,
                                 unsigned char *(*get_output_buffer)(size_t,
                                                                     void *),
                                 void *user);

size_t DP_tile_compress_deflate(DP_Tile *tile, DP_Pixel8 *pixel_buffer,
                                unsigned char *(*get_output_buffer)(size_t,
                                                                    void *),
                                void *user);

size_t DP_tile_compress_split_delta_zstd8le(
    DP_Tile *t, ZSTD_CCtx **in_out_ctx_or_null, DP_SplitTile8 *split_buffer,
    unsigned char *(*get_output_buffer)(size_t, void *), void *user);

size_t DP_tile_compress_mask_delta_zstd8le_opaque(
    unsigned char *(*get_output_buffer)(size_t, void *), void *user);

size_t DP_tile_compress_mask_delta_zstd8le_normal(
    DP_Tile *t, ZSTD_CCtx **in_out_ctx_or_null, uint8_t *channel_buffer,
    unsigned char *(*get_output_buffer)(size_t, void *), void *user);

// Calls *_opaque on tiles where DP_tile_opaque_ident returns true, otherwise
// calls *_normal. For a full opaqueness check, call the above manually.
size_t DP_tile_compress_mask_delta_zstd8le(
    DP_Tile *t, ZSTD_CCtx **in_out_ctx_or_null, uint8_t *channel_buffer,
    unsigned char *(*get_output_buffer)(size_t, void *), void *user);


void DP_tile_copy_to_image(DP_Tile *tile_or_null, DP_Image *img, int x, int y);

void DP_tile_copy_to_pixels8(DP_Tile *tile_or_null, DP_Pixel8 *pixels, int x,
                             int y, int pixels_width, int pixels_height);

void DP_tile_copy_to_upixels8(DP_Tile *tile_or_null, DP_UPixel8 *pixels, int x,
                              int y, int pixels_width, int pixels_height);


void DP_tile_sample(DP_Tile *tile_or_null, const uint16_t *mask, int x, int y,
                    int width, int height, int skip, bool opaque,
                    float *in_out_weight, float *in_out_red,
                    float *in_out_green, float *in_out_blue,
                    float *in_out_alpha);

void DP_tile_sample_pigment(DP_Tile *tile_or_null, const uint16_t *mask, int x,
                            int y, int width, int height, int skip, bool opaque,
                            int sample_interval, float sample_rate,
                            float *in_out_weight, float *in_out_red,
                            float *in_out_green, float *in_out_blue,
                            float *in_out_alpha);


DP_TransientTile *DP_transient_tile_new(DP_Tile *tile, unsigned int context_id);

DP_TransientTile *DP_transient_tile_new_masked(DP_Tile *DP_RESTRICT t,
                                               DP_Tile *DP_RESTRICT mt,
                                               unsigned int context_id);

DP_TransientTile *DP_transient_tile_new_transient(DP_TransientTile *tt,
                                                  unsigned int context_id);

DP_TransientTile *DP_transient_tile_new_blank(unsigned int context_id);

DP_TransientTile *DP_transient_tile_new_nullable(DP_Tile *tile_or_null,
                                                 unsigned int context_id);

DP_TransientTile *DP_transient_tile_new_checker(unsigned int context_id,
                                                DP_Pixel15 pixel1,
                                                DP_Pixel15 pixel2);

DP_TransientTile *DP_transient_tile_incref(DP_TransientTile *tt);

DP_TransientTile *
DP_transient_tile_incref_nullable(DP_TransientTile *tt_or_null);

void DP_transient_tile_decref(DP_TransientTile *tt);

void DP_transient_tile_decref_nullable(DP_TransientTile *tt_or_null);

int DP_transient_tile_refcount(DP_TransientTile *tt);

DP_Tile *DP_transient_tile_persist(DP_TransientTile *tt);


unsigned int DP_transient_tile_context_id(DP_Tile *tt);

DP_Pixel15 *DP_transient_tile_pixels(DP_TransientTile *tt);

DP_Pixel15 DP_transient_tile_pixel_at(DP_TransientTile *tt, int x, int y);

void DP_transient_tile_pixel_at_set(DP_TransientTile *tt, int x, int y,
                                    DP_Pixel15 pixel);

void DP_transient_tile_pixel_at_put(DP_TransientTile *tt, int blend_mode, int x,
                                    int y, DP_Pixel15 pixel);

void DP_transient_tile_clear(DP_TransientTile *tt);

void DP_transient_tile_fill_checker(DP_TransientTile *tt, DP_Pixel15 pixel1,
                                    DP_Pixel15 pixel2);

void DP_transient_tile_copy(DP_TransientTile *tt, DP_Tile *t);

bool DP_transient_tile_blank(DP_TransientTile *tt);

bool DP_transient_tile_opaque(DP_TransientTile *tt);

void DP_transient_tile_mask(DP_TransientTile *DP_RESTRICT tt,
                            DP_Tile *DP_RESTRICT t, DP_Tile *DP_RESTRICT mt);

void DP_transient_tile_mask_in_place(DP_TransientTile *DP_RESTRICT tt,
                                     DP_Tile *DP_RESTRICT mt);

void DP_transient_tile_merge(DP_TransientTile *DP_RESTRICT tt,
                             DP_Tile *DP_RESTRICT t, uint16_t opacity,
                             int blend_mode);

DP_TransientTile *
DP_transient_tile_merge_nullable(DP_TransientTile *DP_RESTRICT tt_or_null,
                                 DP_Tile *DP_RESTRICT t, uint16_t opacity,
                                 int blend_mode);

void DP_transient_tile_brush_apply(DP_TransientTile *tt, DP_UPixel15 src,
                                   int blend_mode, const uint16_t *mask,
                                   uint16_t opacity, int x, int y, int w, int h,
                                   int skip);

void DP_transient_tile_brush_apply_posterize(DP_TransientTile *tt,
                                             int posterize_num,
                                             const uint16_t *mask,
                                             uint16_t opacity, int x, int y,
                                             int w, int h, int skip);

void DP_transient_tile_tint(DP_TransientTile *tt, DP_UPixel8 tint);


#endif
