#!/usr/bin/env python
# SPDX-License-Identifier: GPL-3.0-or-later
import subprocess
import statistics
import sys

bench_executable = sys.argv[1]
total_dabs = 1000
dabs_per_message = 1
runs_to_average = 50

# Generated by running `bench_multidab blendmodes`
blend_modes = [
    {"enum_name": "DP_BLEND_MODE_ERASE", "dptxt_name": "svg:dst-out"},
    {"enum_name": "DP_BLEND_MODE_NORMAL", "dptxt_name": "svg:src-over"},
    {"enum_name": "DP_BLEND_MODE_MULTIPLY", "dptxt_name": "svg:multiply"},
    {"enum_name": "DP_BLEND_MODE_DIVIDE", "dptxt_name": "krita:divide"},
    {"enum_name": "DP_BLEND_MODE_BURN", "dptxt_name": "svg:color-burn"},
    {"enum_name": "DP_BLEND_MODE_DODGE", "dptxt_name": "svg:color-dodge"},
    {"enum_name": "DP_BLEND_MODE_DARKEN", "dptxt_name": "svg:darken"},
    {"enum_name": "DP_BLEND_MODE_LIGHTEN", "dptxt_name": "svg:lighten"},
    {"enum_name": "DP_BLEND_MODE_SUBTRACT", "dptxt_name": "krita:subtract"},
    {"enum_name": "DP_BLEND_MODE_ADD", "dptxt_name": "svg:plus"},
    {"enum_name": "DP_BLEND_MODE_RECOLOR", "dptxt_name": "svg:src-atop"},
    {"enum_name": "DP_BLEND_MODE_BEHIND", "dptxt_name": "svg:dst-over"},
    {"enum_name": "DP_BLEND_MODE_COLOR_ERASE", "dptxt_name": "-dp-cerase"},
    {"enum_name": "DP_BLEND_MODE_SCREEN", "dptxt_name": "svg:screen"},
    {
        "enum_name": "DP_BLEND_MODE_NORMAL_AND_ERASER",
        "dptxt_name": "-dp-normal-and-eraser",
    },
    {
        "enum_name": "DP_BLEND_MODE_LUMINOSITY_SHINE_SAI",
        "dptxt_name": "krita:luminosity_sai",
    },
    {"enum_name": "DP_BLEND_MODE_OVERLAY", "dptxt_name": "svg:overlay"},
    {"enum_name": "DP_BLEND_MODE_HARD_LIGHT", "dptxt_name": "svg:hard-light"},
    {"enum_name": "DP_BLEND_MODE_SOFT_LIGHT", "dptxt_name": "svg:soft-light"},
    {"enum_name": "DP_BLEND_MODE_LINEAR_BURN", "dptxt_name": "krita:linear_burn"},
    {"enum_name": "DP_BLEND_MODE_LINEAR_LIGHT", "dptxt_name": "krita:linear light"},
    {"enum_name": "DP_BLEND_MODE_HUE", "dptxt_name": "svg:hue"},
    {"enum_name": "DP_BLEND_MODE_SATURATION", "dptxt_name": "svg:saturation"},
    {"enum_name": "DP_BLEND_MODE_LUMINOSITY", "dptxt_name": "svg:luminosity"},
    {"enum_name": "DP_BLEND_MODE_COLOR", "dptxt_name": "svg:color"},
    {
        "enum_name": "DP_BLEND_MODE_COMPARE_DENSITY_SOFT",
        "dptxt_name": "-dp-compare-density-soft",
    },
    {"enum_name": "DP_BLEND_MODE_COMPARE_DENSITY", "dptxt_name": "-dp-compare-density"},
    {"enum_name": "DP_BLEND_MODE_VIVID_LIGHT", "dptxt_name": "krita:vivid_light"},
    {"enum_name": "DP_BLEND_MODE_PIN_LIGHT", "dptxt_name": "krita:pin_light"},
    {"enum_name": "DP_BLEND_MODE_DIFFERENCE", "dptxt_name": "svg:difference"},
    {"enum_name": "DP_BLEND_MODE_DARKER_COLOR", "dptxt_name": "krita:darker color"},
    {"enum_name": "DP_BLEND_MODE_LIGHTER_COLOR", "dptxt_name": "krita:lighter color"},
    {"enum_name": "DP_BLEND_MODE_SHADE_SAI", "dptxt_name": "krita:shade_sai"},
    {
        "enum_name": "DP_BLEND_MODE_SHADE_SHINE_SAI",
        "dptxt_name": "krita:shade_shine_sai",
    },
    {"enum_name": "DP_BLEND_MODE_BURN_SAI", "dptxt_name": "krita:burn_sai"},
    {"enum_name": "DP_BLEND_MODE_DODGE_SAI", "dptxt_name": "krita:dodge_sai"},
    {"enum_name": "DP_BLEND_MODE_BURN_DODGE_SAI", "dptxt_name": "krita:burn_dodge_sai"},
    {"enum_name": "DP_BLEND_MODE_HARD_MIX_SAI", "dptxt_name": "krita:hard_mix_sai"},
    {"enum_name": "DP_BLEND_MODE_DIFFERENCE_SAI", "dptxt_name": "krita:difference_sai"},
    {"enum_name": "DP_BLEND_MODE_MARKER", "dptxt_name": "-dp-marker"},
    {"enum_name": "DP_BLEND_MODE_MARKER_WASH", "dptxt_name": "-dp-marker-wash"},
    {"enum_name": "DP_BLEND_MODE_GREATER", "dptxt_name": "-dp-greater"},
    {"enum_name": "DP_BLEND_MODE_GREATER_WASH", "dptxt_name": "-dp-greater-wash"},
    {"enum_name": "DP_BLEND_MODE_PIGMENT", "dptxt_name": "-dp-pigment"},
    {"enum_name": "DP_BLEND_MODE_LIGHT_TO_ALPHA", "dptxt_name": "-dp-light-to-alpha"},
    {"enum_name": "DP_BLEND_MODE_DARK_TO_ALPHA", "dptxt_name": "-dp-dark-to-alpha"},
    {"enum_name": "DP_BLEND_MODE_MULTIPLY_ALPHA", "dptxt_name": "-dp-multiply-alpha"},
    {"enum_name": "DP_BLEND_MODE_DIVIDE_ALPHA", "dptxt_name": "-dp-divide-alpha"},
    {"enum_name": "DP_BLEND_MODE_BURN_ALPHA", "dptxt_name": "-dp-burn-alpha"},
    {"enum_name": "DP_BLEND_MODE_DODGE_ALPHA", "dptxt_name": "-dp-dodge-alpha"},
    {"enum_name": "DP_BLEND_MODE_DARKEN_ALPHA", "dptxt_name": "-dp-darken-alpha"},
    {"enum_name": "DP_BLEND_MODE_LIGHTEN_ALPHA", "dptxt_name": "-dp-lighten-alpha"},
    {"enum_name": "DP_BLEND_MODE_SUBTRACT_ALPHA", "dptxt_name": "-dp-subtract-alpha"},
    {"enum_name": "DP_BLEND_MODE_ADD_ALPHA", "dptxt_name": "-dp-add-alpha"},
    {"enum_name": "DP_BLEND_MODE_SCREEN_ALPHA", "dptxt_name": "-dp-screen-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_LUMINOSITY_SHINE_SAI_ALPHA",
        "dptxt_name": "-dp-luminosity-shine-sai-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_OVERLAY_ALPHA", "dptxt_name": "-dp-overlay-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_HARD_LIGHT_ALPHA",
        "dptxt_name": "-dp-hard-light-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_SOFT_LIGHT_ALPHA",
        "dptxt_name": "-dp-soft-light-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_LINEAR_BURN_ALPHA",
        "dptxt_name": "-dp-linear-burn-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_LINEAR_LIGHT_ALPHA",
        "dptxt_name": "-dp-linear-light-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_HUE_ALPHA", "dptxt_name": "-dp-hue-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_SATURATION_ALPHA",
        "dptxt_name": "-dp-saturation-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_LUMINOSITY_ALPHA",
        "dptxt_name": "-dp-luminosity-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_COLOR_ALPHA", "dptxt_name": "-dp-color-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_VIVID_LIGHT_ALPHA",
        "dptxt_name": "-dp-vivid-light-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_PIN_LIGHT_ALPHA", "dptxt_name": "-dp-pin-light-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_DIFFERENCE_ALPHA",
        "dptxt_name": "-dp-difference-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_DARKER_COLOR_ALPHA",
        "dptxt_name": "-dp-darker-color-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_LIGHTER_COLOR_ALPHA",
        "dptxt_name": "-dp-lighter-color-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_SHADE_SAI_ALPHA", "dptxt_name": "-dp-shade-sai-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_SHADE_SHINE_SAI_ALPHA",
        "dptxt_name": "-dp-shade-shine-sai-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_BURN_SAI_ALPHA", "dptxt_name": "-dp-burn-sai-alpha"},
    {"enum_name": "DP_BLEND_MODE_DODGE_SAI_ALPHA", "dptxt_name": "-dp-dodge-sai-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_BURN_DODGE_SAI_ALPHA",
        "dptxt_name": "-dp-burn-dodge-sai-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_HARD_MIX_SAI_ALPHA",
        "dptxt_name": "-dp-hard-mix-sai-alpha",
    },
    {
        "enum_name": "DP_BLEND_MODE_DIFFERENCE_SAI_ALPHA",
        "dptxt_name": "-dp-difference-sai-alpha",
    },
    {"enum_name": "DP_BLEND_MODE_MARKER_ALPHA", "dptxt_name": "-dp-marker-alpha"},
    {
        "enum_name": "DP_BLEND_MODE_MARKER_ALPHA_WASH",
        "dptxt_name": "-dp-marker-alpha-wash",
    },
    {"enum_name": "DP_BLEND_MODE_GREATER_ALPHA", "dptxt_name": "krita:greater"},
    {
        "enum_name": "DP_BLEND_MODE_GREATER_ALPHA_WASH",
        "dptxt_name": "-dp-greater-alpha-wash",
    },
    {"enum_name": "DP_BLEND_MODE_PIGMENT_ALPHA", "dptxt_name": "mypaint:spectral-wgm"},
    {
        "enum_name": "DP_BLEND_MODE_PIGMENT_AND_ERASER",
        "dptxt_name": "-dp-pigment-and-eraser",
    },
    {
        "enum_name": "DP_BLEND_MODE_ERASE_PRESERVE",
        "dptxt_name": "-dp-erase-preserve",
        "alias_for": "DP_BLEND_MODE_ERASE",
    },
    {
        "enum_name": "DP_BLEND_MODE_BEHIND_PRESERVE",
        "dptxt_name": "-dp-behind-preserve",
        "alias_for": "DP_BLEND_MODE_BEHIND",
    },
    {
        "enum_name": "DP_BLEND_MODE_COLOR_ERASE_PRESERVE",
        "dptxt_name": "-dp-cerase-preserve",
        "alias_for": "DP_BLEND_MODE_COLOR_ERASE",
    },
    {
        "enum_name": "DP_BLEND_MODE_LIGHT_TO_ALPHA",
        "dptxt_name": "-dp-light-to-alpha-preserve",
        "alias_for": "DP_BLEND_MODE_LIGHT_TO_ALPHA",
    },
    {
        "enum_name": "DP_BLEND_MODE_DARK_TO_ALPHA_PRESERVE",
        "dptxt_name": "-dp-dark-to-alpha-preserve",
        "alias_for": "DP_BLEND_MODE_DARK_TO_ALPHA",
    },
    {"enum_name": "DP_BLEND_MODE_REPLACE", "dptxt_name": "-dp-replace"},
]


def run_bench(*args):
    sys.stdout.flush()
    result = subprocess.check_output([bench_executable, *map(str, args)], text=True)
    return float(result.strip())


def to_cost(results):
    return statistics.median(results)


def bench_pixel(pixel_type, alpha, blend_mode):
    results = []
    for i in range(runs_to_average):
        results.append(
            run_bench(
                pixel_type, total_dabs, dabs_per_message, alpha, blend_mode, 255, 255
            )
            / total_dabs
        )
    return to_cost(results) / (255.0 * 255.0)


def bench_classic(alpha, blend_mode):
    results = []
    for i in range(runs_to_average):
        results.append(
            run_bench(
                "classic",
                total_dabs,
                dabs_per_message,
                alpha,
                blend_mode,
                65535,
                255,
                127,
            )
            / total_dabs
        )
    return to_cost(results) / (65535.0 * 65535.0)


def bench_mypaint(lock_alpha, colorize, posterize, mode):
    results = []
    for i in range(runs_to_average):
        results.append(
            run_bench(
                "mypaint",
                total_dabs,
                dabs_per_message,
                255,
                65535,
                255,
                127,
                0,
                0,
                lock_alpha,
                colorize,
                posterize,
                mode,
            )
            / total_dabs
        )
    return to_cost(results) / (65535.0 * 65535.0)


def bench_mypaint_blend(alpha, blend_mode):
    results = []
    for i in range(runs_to_average):
        results.append(
            run_bench(
                "mypaintblend",
                total_dabs,
                dabs_per_message,
                alpha,
                blend_mode,
                65535,
                255,
                127,
                0,
                0,
            )
            / total_dabs
        )
    return to_cost(results) / (65535.0 * 65535.0)


def get_aliases_for(enum_name):
    return [bm["enum_name"] for bm in blend_modes if bm.get("alias_for") == enum_name]


def generate_drawpile_cost(brush_type, fn):
    sys.stdout.write(
        f"double DP_dab_cost_{brush_type}(bool indirect, int blend_mode) {{\n"
    )
    sys.stdout.write("    if (indirect) {\n")

    indirect_cost = fn(255, "svg:src-over")
    sys.stdout.write(f"        return {indirect_cost};\n")

    sys.stdout.write("    }\n")
    sys.stdout.write("    else {\n")
    sys.stdout.write("        switch (blend_mode) {\n")

    costs = [indirect_cost]
    for blend_mode in blend_modes:
        if not "alias_for" in blend_mode:
            cost = fn(0, blend_mode["dptxt_name"])
            costs.append(cost)
            enum_name = blend_mode["enum_name"]
            sys.stdout.write(f"        case {enum_name}:\n")
            for alias in get_aliases_for(enum_name):
                sys.stdout.write(f"        case {alias}:\n")
            sys.stdout.write(f"            return {cost};\n")
    sys.stdout.write("        default:\n")
    sys.stdout.write('            DP_debug("Unhandled blend mode %d", blend_mode);\n')
    sys.stdout.write(f"            return {max(costs)};\n")

    sys.stdout.write("        }\n")
    sys.stdout.write("    }\n")
    sys.stdout.write("}\n\n")


def generate_pixel_cost(name, pixel_type):
    generate_drawpile_cost(
        name, lambda alpha, blend_mode: bench_pixel(pixel_type, alpha, blend_mode)
    )


def generate_classic_cost():
    generate_drawpile_cost(
        "classic", lambda alpha, blend_mode: bench_classic(alpha, blend_mode)
    )


def generate_mypaint_cost():
    sys.stdout.write(
        "double DP_dab_cost_mypaint(bool indirect, uint8_t lock_alpha, uint8_t colorize, uint8_t posterize) {\n"
    )
    sys.stdout.write("    if (indirect) {\n")
    sys.stdout.write(f"        return {bench_mypaint(0, 0, 0, 129)};\n")
    sys.stdout.write("    }\n")
    sys.stdout.write("    else {\n")
    sys.stdout.write("        double cost = 0.0;\n")
    sys.stdout.write("        if (lock_alpha != 0) {\n")
    sys.stdout.write(f"            cost += {bench_mypaint(255, 0, 0, 0)};\n")
    sys.stdout.write("        }\n")
    sys.stdout.write("        if (colorize != 0) {\n")
    sys.stdout.write(f"            cost += {bench_mypaint(0, 255, 0, 0)};\n")
    sys.stdout.write("        }\n")
    sys.stdout.write("        if (posterize != 0) {\n")
    sys.stdout.write(f"            cost += {bench_mypaint(0, 0, 255, 127)};\n")
    sys.stdout.write("        }\n")
    sys.stdout.write(
        "        if (lock_alpha != UINT8_MAX && colorize != UINT8_MAX && posterize != UINT8_MAX) {\n"
    )
    sys.stdout.write(f"            cost += {bench_mypaint(0, 0, 0, 0)};\n")
    sys.stdout.write("        }\n")
    sys.stdout.write("        return cost;\n")
    sys.stdout.write("    }\n")
    sys.stdout.write("}\n\n")


def generate_mypaint_blend_cost():
    generate_drawpile_cost(
        "mypaint_blend",
        lambda alpha, blend_mode: bench_mypaint_blend(alpha, blend_mode),
    )


sys.stdout.write("// SPDX-License-Identifier: GPL-3.0-or-later\n")
sys.stdout.write("// Auto-generated by generate_dab_cost.py\n")
sys.stdout.write('#include "dab_cost.h"\n')
sys.stdout.write("#include <dpcommon/common.h>\n")
sys.stdout.write("#include <dpmsg/blend_mode.h>\n")
sys.stdout.write("\n")

generate_pixel_cost("pixel", "pixelround")
generate_pixel_cost("pixel_square", "pixelsquare")
generate_classic_cost()
generate_mypaint_cost()
generate_mypaint_blend_cost()
