package net.damschen.swatchit.test.ui.viewmodels

import android.net.Uri
import androidx.core.net.toUri
import androidx.test.ext.junit.runners.AndroidJUnit4
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.test.UnconfinedTestDispatcher
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.MeasurementType
import net.damschen.swatchit.domain.aggregates.swatch.Name
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.shared.testhelpers.testdata.SwatchTestData
import net.damschen.swatchit.test.testHelpers.FakePhotoStorageService
import net.damschen.swatchit.test.testHelpers.MainDispatcherRule
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.PhotoState
import net.damschen.swatchit.ui.models.SwatchFormState
import net.damschen.swatchit.ui.models.SwatchState
import net.damschen.swatchit.ui.models.ValidatedInput
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.Assert.assertEquals
import org.junit.Assert.assertFalse
import org.junit.Assert.assertNotNull
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import java.time.LocalDate
import java.time.ZoneId
import java.util.UUID

@RunWith(AndroidJUnit4::class)
class EditSwatchViewModelTests {
    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    private var repo = FakeRepo()
    private var photoStorageService = FakePhotoStorageService()
    private lateinit var sut: EditSwatchViewModel


    @Before
    fun initSut() {
        repo = FakeRepo()
        sut = EditSwatchViewModel(
            repo.defaultId,
            repo,
            FakeDateTimeProvider(),
            photoStorageService,
            FakeUUIDProvider()
        )
    }

    @Test
    fun loadSwatch_RepositoryReturnsSuccess_loadsSwatch() {
        assertTrue(sut.loadState.value is LoadState.Success)
        val swatchState = SwatchState.from(repo.swatchToReturn())!!
        assertEquals(
            SwatchFormState.fromSwatchState(swatchState),
            sut.formManager.swatchFormState.value
        )
        assertEquals(repo.defaultGauge, sut.gaugeFormState.value.toGauge())
    }

    @Test
    fun loadSwatch_RepositoryReturnsSuccessWIthNullData_setsNotFoundLoadState() {
        repo.returnNull = true
        sut = EditSwatchViewModel(
            repo.defaultId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )

        assertTrue(sut.loadState.value is LoadState.NotFound)
    }

    @Test
    fun loadSwatch_RepositoryReturnsError_setsErrorLoadState() {
        repo.returnError = true
        sut = EditSwatchViewModel(
            repo.defaultId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )

        assertTrue(sut.loadState.value is LoadState.Error)
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveSwatchForm_RepositoryReturnsSuccess_emitsSuccess() = runTest {
        val pattern = "new Pattern"

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.swatchFormSuccessfullySaved.collect {
                values.add(it)
            }
        }
        sut.formManager.onPatternChange(pattern)

        sut.saveSwatchForm()

        assertTrue(values.last())
        assertEquals(pattern, repo.updatedSwatch!!.pattern!!.value)
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveSwatchForm_RepositoryReturnsError_emitsError() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.swatchFormSuccessfullySaved.collect {
                values.add(it)
            }
        }
        repo.returnError = true

        sut.saveSwatchForm()

        assertFalse(values.last())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveSwatchForm_invalidInput_emitsError() = runTest {
        sut.formManager.onNameChange("A".repeat(51))

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.swatchFormSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveSwatchForm()

        assertFalse(values.last())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun deleteSwatch_RepositoryReturnsSuccess_emitsSuccess() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.successfullyDeleted.collect {
                values.add(it)
            }
        }

        sut.deleteSwatch()

        assertTrue(values.last())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun deleteSwatch_RepositoryReturnsError_emitsError() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.successfullyDeleted.collect {
                values.add(it)
            }
        }
        repo.returnError = true

        sut.deleteSwatch()

        assertFalse(values.last())
    }

    @Test
    fun onNrOfStitchesChange_validInput_updatesNrOfStitches() {
        val validCount = "20"

        sut.onNrOfStitchesChange(validCount)

        val stitchesState = sut.gaugeFormState.value.nrOfStitches
        assertEquals(validCount, stitchesState.value)
        assertTrue(stitchesState is ValidatedInput.Valid)
    }

    @Test
    fun onNrOfStitchesChange_invalidInput_updatesNrOfStitchesWithError() {
        val invalidCount = "invalid"

        sut.onNrOfStitchesChange(invalidCount)

        val stitchesState = sut.gaugeFormState.value.nrOfStitches
        assertEquals(invalidCount, stitchesState.value)
        assertFalse(stitchesState is ValidatedInput.Valid)
    }

    @Test
    fun onNrOfRowsChange_validInput_updatesNrOfRows() {
        val validCount = "25"

        sut.onNrOfRowsChange(validCount)

        val rowsState = sut.gaugeFormState.value.nrOfRows
        assertEquals(validCount, rowsState.value)
        assertTrue(rowsState is ValidatedInput.Valid)
    }

    @Test
    fun onNrOfRowsChange_invalidInput_updatesNrOfRowsWithError() {
        val invalidCount = "invalid"

        sut.onNrOfRowsChange(invalidCount)

        val rowsState = sut.gaugeFormState.value.nrOfRows
        assertEquals(invalidCount, rowsState.value)
        assertFalse(rowsState is ValidatedInput.Valid)
    }

    @Test
    fun onGaugeLengthChange_validInput_updatesGaugeLength() {
        val validLength = "10.5"

        sut.onGaugeLengthChange(validLength)

        val gaugeLengthState = sut.gaugeFormState.value.size
        assertEquals(validLength, gaugeLengthState.value)
        assertTrue(gaugeLengthState is ValidatedInput.Valid)
    }

    @Test
    fun onGaugeLengthChange_invalidInput_updatesGaugeLengthWithError() {
        val invalidLength = "invalid"

        sut.onGaugeLengthChange(invalidLength)

        val gaugeLengthState = sut.gaugeFormState.value.size
        assertEquals(invalidLength, gaugeLengthState.value)
        assertFalse(gaugeLengthState is ValidatedInput.Valid)
    }

    @Test
    fun saveGaugeForm_repoReturnsSuccess_callsRepo() {
        val initialSwatch = SwatchTestData.from(repo.swatchToReturn())!!

        sut.onNrOfStitchesChange("13")
        sut.onNrOfRowsChange("23")
        sut.onGaugeLengthChange("33")
        sut.saveGaugeForm()

        val expected = initialSwatch.copy(
            gauge = Gauge(
                GaugeCount(13), GaugeCount(23),
                GaugeSize(33.0)
            )
        )

        assertEquals(expected, SwatchTestData.from(repo.updatedSwatch))
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeForm_repoReturnsSuccess_emitSuccess() = runTest {
        sut.onNrOfStitchesChange("13")
        sut.onNrOfRowsChange("23")
        sut.onGaugeLengthChange("33")

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeFormSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveGaugeForm()

        assertTrue(values.last())
    }

    @Test
    fun loadSwatch_gaugeIsNull_errorIdsSetInGaugeState() = runTest {
        repo.swatchToReturn = {
            Swatch.create(
                needleSize = KnittingNeedleSize.SIZE_10_0,
                pattern = null,
                yarn = null,
                notes = null,
                createdAt = EpochMillis(0),
                id = SwatchId(1),
                name = null
            )
        }
        sut = EditSwatchViewModel(
            repo.defaultId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )

        assertNotNull((sut.gaugeFormState.value.nrOfStitches as ValidatedInput.Invalid).errorMessageId)
        assertNotNull((sut.gaugeFormState.value.nrOfRows as ValidatedInput.Invalid).errorMessageId)
        assertNotNull((sut.gaugeFormState.value.size as ValidatedInput.Invalid).errorMessageId)
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeForm_invalidInput_setsError() = runTest {
        sut.onNrOfStitchesChange("13")
        sut.onNrOfRowsChange("23")
        sut.onGaugeLengthChange("-33")

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeFormSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveGaugeForm()

        assertFalse(values.last())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeForm_repoReturnsErrorDuringUpdate_setsErrorState() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeFormSuccessfullySaved.collect {
                values.add(it)
            }
        }

        repo.returnErrorDuringUpdate = true

        sut.saveGaugeForm()

        assertFalse(values.last())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeForm_swatchNotFound_setsErrorState() = runTest {
        repo.swatchToReturn = { null }
        sut = EditSwatchViewModel(
            repo.defaultId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeFormSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveGaugeForm()

        assertFalse(values.last())
    }

    @Test
    fun updateSwatchPhoto_callsRepoWithPhoto() {
        sut.updateSwatchPhoto(defaultUri)
        val updatedSwatch = SwatchTestData.from(repo.swatchToReturn())!!.copy(
            photo = Photo(
                FakeUUIDProvider.defaultUUID
            )
        )

        assertEquals(updatedSwatch, SwatchTestData.from(repo.updatedSwatch))
    }

    @Test
    fun updateSwatchPhoto_updateSuccessful_updatesState() {
        sut.updateSwatchPhoto(defaultUri)

        assertEquals(PhotoState("${FakeUUIDProvider.defaultUUID}.jpg", true), sut.photoState.value)
    }

    @Test
    fun updateSwatchPhoto_emptyUri_doesNotUpdateAndSetsErrorState() {
        sut.updateSwatchPhoto(Uri.EMPTY)

        assertEquals(PhotoState(null, false), sut.photoState.value)
    }

    @Test
    fun updateSwatchPhoto_successfulAfterFailure_setsCorrectState() {
        sut.updateSwatchPhoto(Uri.EMPTY)

        sut.updateSwatchPhoto(defaultUri)

        assertEquals(PhotoState("${FakeUUIDProvider.defaultUUID}.jpg", true), sut.photoState.value)
    }

    @Test
    fun updateSwatchPhoto_photoStorageServiceReturnsErrorDuringCopy_setsErrorState() {
        photoStorageService.returnError = true

        sut.updateSwatchPhoto(defaultUri)

        assertEquals(PhotoState(null, false), sut.photoState.value)
    }

    @Test
    fun updateSwatchPhoto_repoReturnsError_setsErrorState() {
        repo.returnErrorDuringUpdate = true

        sut.updateSwatchPhoto(defaultUri)

        assertEquals(PhotoState(null, false), sut.photoState.value)
    }

    @Test
    fun updateSwatchPhoto_swatchHasPhoto_callsDeleteAndUpdates() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()

        sut.updateSwatchPhoto(defaultUri)

        assertEquals(defaultPhoto, photoStorageService.deletedPhoto)
    }

    @Test
    fun updateSwatchPhoto_repoReturnsError_deletesAddedPhoto() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()

        repo.returnErrorDuringUpdate = true

        sut.updateSwatchPhoto(defaultUri)

        assertEquals(Photo(FakeUUIDProvider.defaultUUID), photoStorageService.deletedPhoto)
    }

    @Test
    fun updateSwatchPhoto_photoStorageServiceReturnsErrorDuringDelete_ignoresError() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()

        photoStorageService.returnErrorDuringDelete = true

        sut.updateSwatchPhoto(defaultUri)

        assertEquals(PhotoState("${FakeUUIDProvider.defaultUUID}.jpg", true), sut.photoState.value)
    }

    @Test
    fun deleteSwatchPhoto_callsRepoWithoutPhoto() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        sut.deleteSwatchPhoto()
        val updatedSwatch = SwatchTestData.from(repo.swatchToReturn())!!.copy(
            photo = null
        )

        assertEquals(updatedSwatch, SwatchTestData.from(repo.updatedSwatch))
    }

    @Test
    fun deleteSwatchPhoto_deleteSuccessful_updatesState() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        sut.deleteSwatchPhoto()

        assertEquals(PhotoState(null, true), sut.photoState.value)
    }

    @Test
    fun deleteSwatchPhoto_callsDeleteWithPhoto() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        sut.deleteSwatchPhoto()

        assertEquals(defaultPhoto, photoStorageService.deletedPhoto)
    }

    @Test
    fun deleteSwatchPhoto_swatchWithoutSwatchPhoto_doesNotCallUpdate() {
        sut.deleteSwatchPhoto()

        assertEquals(null, SwatchTestData.from(repo.updatedSwatch))
    }

    @Test
    fun deleteSwatchPhoto_swatchWithoutSwatchPhoto_doesNotUpdateState() {
        sut.deleteSwatchPhoto()

        assertEquals(PhotoState(null, true), sut.photoState.value)
    }

    @Test
    fun deleteSwatchPhoto_photoStorageServiceReturnsError_setsErrorState() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        photoStorageService.returnError = true

        sut.deleteSwatchPhoto()

        assertEquals(PhotoState(defaultPhoto.fileName, false), sut.photoState.value)
    }

    @Test
    fun deleteSwatchPhoto_successfulAfterFailure_setsCorrectState() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        photoStorageService.returnError = true

        sut.deleteSwatchPhoto()
        photoStorageService.returnError = false

        sut.deleteSwatchPhoto()

        assertEquals(PhotoState(null, true), sut.photoState.value)
    }

    @Test
    fun deleteSwatchPhoto_repoReturnsError_setsErrorState() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        repo.returnErrorDuringUpdate = true

        sut.deleteSwatchPhoto()

        assertEquals(PhotoState(defaultPhoto.fileName, false), sut.photoState.value)
    }

    @Test
    fun deleteSwatch_swatchWithPhoto_callsPhotoStorageServiceDeleteWithPhoto() {
        repo.swatchToReturn = swatchWithPhoto
        sut.loadSwatch()
        sut.deleteSwatch()

        assertEquals(defaultPhoto, photoStorageService.deletedPhoto)
    }

    @Test
    fun deleteSwatch_validSwatch_callsRepoWithId() {
        sut.deleteSwatch()

        assertEquals(repo.defaultId, repo.deletedId?.value)
    }
}

private val defaultNeedleSize = KnittingNeedleSize.SIZE_2_5
const val defaultId = 1
private val defaultPattern = Pattern.create("Stockinette")
val defaultGauge =
    Gauge(GaugeCount(20), GaugeCount(30), GaugeSize(10.0))
private val defaultYarn =
    Yarn.create(Name.create("Yarn Name"), Name.create("Yarn Manufacturer"))
private val defaultNotes = Notes.create("Test notes!")
private val defaultCreatedAt = EpochMillis(
    LocalDate.of(2025, 2, 17).atStartOfDay(
        ZoneId.of("UTC")
    ).toInstant().toEpochMilli()
)
private val defaultName = Name.create("Fake Name")

private val defaultPhoto = Photo(UUID.fromString("bd0ab487-c374-41b6-a7f8-d5390ef94551"))

private val defaultMeasurement =
    Measurement(GaugeCount(34), GaugeSize(13.0), MeasurementType.Rows)

private var swatchWithPhoto = {
    Swatch.create(
        needleSize = defaultNeedleSize,
        pattern = defaultPattern,
        yarn = defaultYarn,
        notes = defaultNotes,
        createdAt = defaultCreatedAt,
        id = SwatchId(defaultId),
        name = defaultName
    ).withUpdatedGauge(defaultGauge).withMeasurement(defaultMeasurement)
        .withUpdatedPhoto(defaultPhoto)
}
private val defaultUri = "http://www.asdf.de".toUri()