package net.damschen.swatchit.test.ui.screens

import android.content.Context
import android.graphics.Bitmap
import androidx.activity.ComponentActivity
import androidx.compose.ui.test.assertIsDisplayed
import androidx.compose.ui.test.assertIsEnabled
import androidx.compose.ui.test.assertIsNotDisplayed
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.isDisplayed
import androidx.compose.ui.test.junit4.createAndroidComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.core.net.toUri
import androidx.test.core.app.ApplicationProvider
import junit.framework.TestCase
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.infrastructure.services.AndroidPhotoStorageService
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.shared.testhelpers.FakeBitmapDecoder
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.screens.SwatchDetailsScreen
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.annotation.Config
import org.robolectric.shadows.ShadowLog
import java.io.File
import java.io.FileOutputStream

@RunWith(RobolectricTestRunner::class)
@Config(sdk = [33], manifest = Config.NONE)
class SwatchDetailsScreenFileSystemTests {
    @get:Rule
    val composeTestRule = createAndroidComposeRule<ComponentActivity>()

    private lateinit var viewModel: EditSwatchViewModel
    private var itemId: Int = 1
    private lateinit var defaultSwatch: Swatch
    private var repo = FakeRepo()

    private lateinit var context: Context
    private lateinit var photoStorageService: PhotoStorageService

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initViewModel() {
        context = ApplicationProvider.getApplicationContext()
        photoStorageService = AndroidPhotoStorageService(context, FakeBitmapDecoder())
        repo = FakeRepo()
        defaultSwatch = repo.swatchToReturn()!!
        itemId = repo.defaultId
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            photoStorageService,
            FakeUUIDProvider()
        )
    }

    @Test
    fun photo_swatchWithPhoto_showsPhotoCard() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchPhotoCaption").assertExists()
    }

    @Test
    fun photo_swatchWithoutPhoto_showsPickPhotoButton() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("PhotoPickerButton").assertExists()
    }

    @Test
    fun photo_swatchWithoutPhoto_deletePhotoButtonIsNotEnabled() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("DeletePhotoButton").assertIsNotEnabled()
    }

    @Test
    fun photo_swatchWithPhoto_deletePhotoButtonIsEnabled() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.onNodeWithTag("DeletePhotoButton").assertIsEnabled()
    }

    @Test
    fun photo_swatchWithPhoto_showsPhoto() = runTest {
        createTestImageFile(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchBitmap").assertExists()
    }

    @Test
    fun photo_swatchWithoutPhoto_doesNotShowPhoto() = runTest {
        createTestImageFile(context)
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchBitmap").performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun photo_swatchWithPhotoNotFoundInFileSystem_doesNotShowPhoto() = runTest {
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchBitmap").performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun photo_swatchWithPhotoNotFoundInFileSystem_showsErrorMessage() = runTest {
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("BitmapNotFoundError").assertExists()
    }

    @Test
    fun deletePhoto_swatchWithPhoto_removesPhoto() = runTest {
        createTestImageFile(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("DeletePhotoButton").performScrollTo().performClick()
        composeTestRule.waitUntil {
            composeTestRule.onNodeWithTag("ConfirmDeleteDialog").isDisplayed()
        }
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        composeTestRule.onNodeWithTag("SwatchBitmap").performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun deletePhoto_swatchWithPhotoCancelDeletion_doesNotRemovePhoto() = runTest {
        createTestImageFile(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("DeletePhotoButton").performScrollTo().performClick()
        composeTestRule.waitUntil {
            composeTestRule.onNodeWithTag("ConfirmDeleteDialog").isDisplayed()
        }
        composeTestRule.onNodeWithTag("CancelDeleteButton").performClick()

        composeTestRule.onNodeWithTag("SwatchBitmap").assertExists()
    }

    @Test
    fun photoPicker_errorDuringUpdate_showsErrorMessage() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        repo.returnErrorDuringUpdate = true
        val testUri = "content://media/external/images/media/1".toUri()
        viewModel.updateSwatchPhoto(testUri)

        composeTestRule.onNodeWithTag("PhotoUpdateError").performScrollTo().assertIsDisplayed()
    }

    @Test
    fun deletePhoto_errorDuringUpdate_showsErrorMessage() = runTest {
        createTestImageFile(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        repo.returnErrorDuringUpdate = true
        composeTestRule.onNodeWithTag("DeletePhotoButton").performScrollTo().performClick()
        composeTestRule.waitUntil {
            composeTestRule.onNodeWithTag("ConfirmDeleteDialog").isDisplayed()
        }
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        composeTestRule.onNodeWithTag("PhotoUpdateError").performScrollTo().assertIsDisplayed()
        composeTestRule.onNodeWithTag("BitmapNotFoundError").assertExists()
    }

    @Test
    fun deleteSwatch_validInput_removesSwatchFromDb() = runTest {
        createTestImageFile(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag("OptionsMenu").performClick()
        composeTestRule.onNodeWithTag("DeleteButton").performClick()
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        val savedFile = File(context.filesDir, "images/${FakeUUIDProvider.defaultUUID}.jpg")
        TestCase.assertFalse(savedFile.exists())
    }
}

private fun defaultSwatchWithoutGauge() = Swatch.create(
    needleSize = KnittingNeedleSize.SIZE_10_0,
    pattern = null,
    yarn = null,
    notes = null,
    createdAt = EpochMillis(0),
    id = SwatchId(1),
    name = null
)

private fun createTestImageFile(context: Context) {
    val bitmap = Bitmap.createBitmap(50, 50, Bitmap.Config.ARGB_8888)
    bitmap.eraseColor(android.graphics.Color.BLUE)
    val dir = File(context.filesDir, "images").apply { mkdirs() }

    val file = File(dir, "${FakeUUIDProvider.defaultUUID}.jpg")
    FileOutputStream(file).use { out ->
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, out)
    }
    bitmap.recycle()
}