package net.damschen.swatchit.test.ui.screens

import androidx.compose.ui.test.assert
import androidx.compose.ui.test.assertCountEquals
import androidx.compose.ui.test.assertIsDisplayed
import androidx.compose.ui.test.assertIsEnabled
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.assertTextContains
import androidx.compose.ui.test.hasText
import androidx.compose.ui.test.isNotDisplayed
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onChildren
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.onNodeWithText
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.compose.ui.test.performTextInput
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.MeasurementType
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.enums.CountType
import net.damschen.swatchit.ui.screens.MeasurementsScreen
import net.damschen.swatchit.ui.viewmodels.MeasurementViewModel
import org.junit.Assert.assertEquals
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog

@RunWith(RobolectricTestRunner::class)
class MeasurementsScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    private lateinit var viewModel: MeasurementViewModel
    private var repository = FakeRepo()


    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
        viewModel = MeasurementViewModel(repository.defaultId, repository)
    }

    @Test
    fun backButtonClicked_callsOnNavigateBack() = runTest {
        var backClicked = false
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) { backClicked = true }
        }

        composeTestRule.onNodeWithTag("BackButton").performClick()

        assertTrue(backClicked)
    }


    @Test
    fun addDialog_invalidInput_addButtonDisabled() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("AddMeasurementButton").assertIsNotEnabled()
    }

    @Test
    fun addDialog_validInput_addButtonEnabled() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("AddButton").performClick()
        composeTestRule.onNodeWithTag("Count").performTextInput("25")
        composeTestRule.onNodeWithTag("Size").performTextInput("17")
        composeTestRule.onNodeWithTag("CountType").performClick()
        composeTestRule.onNodeWithText(CountType.Rows.name).performClick()

        composeTestRule.onNodeWithTag("AddMeasurementButton").assertIsEnabled()
    }

    @Test
    fun addButtonClicked_showsDialog() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("AddMeasurementDialog").assertIsDisplayed()
    }

    @Test
    fun addDialog_repositoryReturnsError_showsError() {
        repository.returnError = true
        viewModel = MeasurementViewModel(repository.defaultId, repository)

        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("Error").assertExists()
    }

    @Test
    fun addDialog_cancelButtonClicked_closesDialog() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("CancelAddMeasurementButton").performScrollTo().performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("AddMeasurementDialog").isNotDisplayed()
        }
        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)
    }

    @Test
    fun addDialog_cancelButtonClicked_resetsState() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("AddButton").performClick()
        composeTestRule.onNodeWithTag("Count").performTextInput("25")
        composeTestRule.onNodeWithTag("Size").performTextInput("17")
        composeTestRule.onNodeWithTag("CountType").performClick()
        composeTestRule.onNodeWithText(CountType.Rows.name).performClick()

        composeTestRule.onNodeWithTag("CancelAddMeasurementButton").performScrollTo().performClick()

        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("Count").assert(hasText(""))
        composeTestRule.onNodeWithTag("Size").assert(hasText(""))
        composeTestRule.onNodeWithTag("CountType").assert(hasText(CountType.Stitches.name))
    }

    @Test
    fun addDialog_addReturnsError_errorIsShownToUser() {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }
        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("Count").performTextInput("25")
        composeTestRule.onNodeWithTag("Size").performTextInput("17")
        composeTestRule.onNodeWithTag("CountType").performClick()
        composeTestRule.onNodeWithText(CountType.Rows.name).performClick()

        repository.returnErrorDuringUpdate = true
        composeTestRule.onNodeWithTag("AddMeasurementButton").performClick()
        composeTestRule.onNodeWithTag("AddMeasurementDialog").assertIsDisplayed()

        composeTestRule.onNodeWithTag("Error").assertExists()
    }

    @Test
    fun deleteReturnsError_errorMessageIsShown() {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }
        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)

        composeTestRule.onNodeWithTag("optionsMenu_0").performClick()
        composeTestRule.onNodeWithTag("DeleteButton").performClick()
        repository.returnError = true
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("ConfirmButton").isNotDisplayed()
        }
        composeTestRule.onNode(
            hasText("Error while deleting", substring = true), useUnmergedTree = true
        ).assertIsDisplayed()
    }

    @Test
    fun calculateGaugeButtonClicked_showsDialog() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()

        composeTestRule.onNodeWithTag("CalculateGaugeDialog").assertIsDisplayed()
    }

    @Test
    fun calculateGaugeDialog_cancelButtonClicked_closesDialog() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()

        composeTestRule.onNodeWithTag("CancelCalculateGaugeButton").performScrollTo().performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("CalculateGaugeDialog").isNotDisplayed()
        }
    }

    @Test
    fun calculateGaugeDialog_noWidthMeasurement_showsError() = runTest {
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()

        composeTestRule.onNodeWithTag("CalculateGaugeErrorMessage").assertIsDisplayed()
    }

    @Test
    fun calculateGaugeDialog_validMeasurement_showsResults() = runTest {
        val swatch = repository.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repository.swatchToReturn = { swatch }
        viewModel = MeasurementViewModel(repository.defaultId, repository)
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()

        composeTestRule.onNodeWithTag("GaugeCalculationResult")
            .assertTextContains("34 stitches x 34 rows\n≙\n10 cm x 10 cm")
    }

    @Test
    fun calculateGaugeDialog_saveClicked_callsRepoAndClosesDialog() = runTest {
        val swatch = repository.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repository.swatchToReturn = { swatch }
        viewModel = MeasurementViewModel(repository.defaultId, repository)
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()
        composeTestRule.onNodeWithTag("SaveGaugeCalculation").performClick()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("CalculateGaugeDialog").isNotDisplayed()
        }
        assertEquals(
            Gauge(GaugeCount(34), GaugeCount(34), GaugeSize(10.0)),
            repository.updatedSwatch!!.gauge
        )
    }

    @Test
    fun calculateGaugeDialog_saveClickedRepoReturnsError_showErrorInDialog() = runTest {
        val swatch = repository.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repository.swatchToReturn = { swatch }
        viewModel = MeasurementViewModel(repository.defaultId, repository)
        composeTestRule.setContent {
            MeasurementsScreen(
                repository.defaultId, viewModel
            ) {}
        }

        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()
        repository.returnError = true
        composeTestRule.onNodeWithTag("SaveGaugeCalculation").performClick()

        composeTestRule.onNodeWithTag("Error").assertIsDisplayed()
    }
}
