package net.damschen.swatchit.test.ui.screens

import androidx.compose.ui.test.assertIsEnabled
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.compose.ui.test.performTextInput
import androidx.compose.ui.test.performTextReplacement
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.test.testHelpers.FakePhotoStorageService
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.screens.EditSwatchScreen
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog

@RunWith(RobolectricTestRunner::class)
class EditSwatchScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    private lateinit var viewModel: EditSwatchViewModel
    private var itemId: Int = 1

    private var repo: FakeRepo = FakeRepo()

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initViewModel() {
        itemId = repo.defaultId
        viewModel = EditSwatchViewModel(
            itemId, repo, FakeDateTimeProvider(),
            FakePhotoStorageService(), FakeUUIDProvider()
        )
    }

    @Test
    fun backButtonClicked_callsOnDismiss() {
        var dismissed = false

        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = { dismissed = true })
        }

        composeTestRule.onNodeWithTag("BackButton").performClick()

        assertTrue(dismissed)
    }

    @Test
    fun nameInvalid_saveChangesButtonIsDisabled() {
        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("Name").performScrollTo().performClick()
            .performTextInput("A".repeat(51))

        composeTestRule.onNodeWithTag(saveChangesButtonTag).assertIsNotEnabled()
    }

    @Test
    fun validChanges_saveChangesButtonEnabled() {
        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = {})
        }
        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)

        composeTestRule.onNodeWithTag(saveChangesButtonTag).assertIsEnabled()
    }

    @Test
    fun repositoryReturnsErrorDuringUpdate_showsErrorDialog() {
        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = {})
        }

        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)
        repo.returnError = true
        composeTestRule.onNodeWithTag(saveChangesButtonTag).performClick()

        composeTestRule.onNodeWithTag(updateErrorDialogTag).assertExists()
    }

    @Test
    fun updateErrorDialog_cancelClosesDialog() {
        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = {})
        }

        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)
        repo.returnError = true
        composeTestRule.onNodeWithTag(saveChangesButtonTag).performClick()

        composeTestRule.onNodeWithTag(updateErrorDialogTag).assertExists()

        composeTestRule.onNodeWithTag("CancelErrorDialogButton").performClick()

        composeTestRule.onNodeWithTag(updateErrorDialogTag).assertDoesNotExist()
    }

    @Test
    fun updateErrorDialog_retryButtonTriggersRetry() {
        var onNavigateBackCalled = false
        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = { onNavigateBackCalled = true })
        }

        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)
        repo.returnError = true
        composeTestRule.onNodeWithTag(saveChangesButtonTag).performClick()

        composeTestRule.onNodeWithTag(updateErrorDialogTag).assertExists()
        repo.returnError = false
        composeTestRule.onNodeWithTag("RetryErrorDialogButton").performClick()

        assertTrue(onNavigateBackCalled)
    }

    @Test
    fun updateSwatch_callsOnNavigateBack() {
        var onNavigateBackCalled = false
        composeTestRule.setContent {
            EditSwatchScreen(viewModel, onNavigateBack = { onNavigateBackCalled = true })
        }

        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)
        composeTestRule.onNodeWithTag(saveChangesButtonTag).performClick()

        assertTrue(onNavigateBackCalled)
    }
}

private const val saveChangesButtonTag = "SaveChangesButton"
private const val patternInputTag = "Pattern"
private const val updateErrorDialogTag = "UpdateErrorDialog"