package net.damschen.swatchit.test.testHelpers.database

import android.database.sqlite.SQLiteException
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.MeasurementType
import net.damschen.swatchit.domain.aggregates.swatch.Name
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.repositories.SwatchRepository
import java.time.LocalDate
import java.time.ZoneId

internal class FakeRepo : SwatchRepository {

    private val defaultNeedleSize = KnittingNeedleSize.SIZE_2_5
    val defaultId = 1
    private val defaultPattern = Pattern.create("Stockinette")
    val defaultGauge =
        Gauge(GaugeCount(20), GaugeCount(30), GaugeSize(10.0))
    private val defaultYarn =
        Yarn.create(Name.create("Yarn Name"), Name.create("Yarn Manufacturer"))
    private val defaultNotes = Notes.create("Test notes!")
    private val defaultCreatedAt = EpochMillis(
        LocalDate.of(2025, 2, 17).atStartOfDay(
            ZoneId.of("UTC")
        ).toInstant().toEpochMilli()
    )
    private val defaultName = Name.create("Fake Name")

    val defaultMeasurement =
        Measurement(GaugeCount(34), GaugeSize(13.0), MeasurementType.Rows)

    var swatchToReturn: () -> Swatch? = {
        Swatch.create(
            needleSize = defaultNeedleSize,
            pattern = defaultPattern,
            yarn = defaultYarn,
            notes = defaultNotes,
            createdAt = defaultCreatedAt,
            id = SwatchId(defaultId),
            name = defaultName
        ).withUpdatedGauge(defaultGauge).withMeasurement(defaultMeasurement)
    }

    var returnNull = false

    var returnError = false

    var returnErrorDuringUpdate = false

    var updatedSwatch: Swatch? = null

    var deletedId: SwatchId? = null

    override fun getAll(): Flow<DatabaseResult<List<Swatch>>> = flow {
        if (returnError) emit(DatabaseResult.Error(SQLiteException("Error")))
        else {
            emit(DatabaseResult.Success(listOf(swatchToReturn()!!)))
        }
    }

    override suspend fun get(id: Int): DatabaseResult<Swatch?> {
        if (returnNull) return DatabaseResult.Success(null)
        if (returnError) return DatabaseResult.Error(SQLiteException("Error"))
        return DatabaseResult.Success(swatchToReturn())
    }

    override suspend fun add(swatch: Swatch): DatabaseResult<Unit> =
        DatabaseResult.Success(Unit).takeUnless { returnError } ?: DatabaseResult.Error(
            SQLiteException("Error")
        )

    override suspend fun update(swatch: Swatch): DatabaseResult<Unit> {
        updatedSwatch = swatch
        return DatabaseResult.Success(Unit).takeUnless { returnError || returnErrorDuringUpdate }
            ?: DatabaseResult.Error(
                SQLiteException("Error")
            )
    }

    override suspend fun delete(id: SwatchId): DatabaseResult<Unit> {
        deletedId = id
        return DatabaseResult.Success(Unit).takeUnless { returnError } ?: DatabaseResult.Error(
            SQLiteException("Error")
        )
    }
}