package net.damschen.swatchit.test.domain.aggregates.swatch

import junit.framework.TestCase.assertEquals
import junit.framework.TestCase.assertNull
import net.damschen.swatchit.domain.aggregates.swatch.Count
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Size
import org.junit.Test
import org.junit.experimental.theories.DataPoints
import org.junit.experimental.theories.Theories
import org.junit.experimental.theories.Theory
import org.junit.runner.RunWith

@RunWith(Theories::class)
class GaugeTests {
    @Test
    fun init_allParametersSet_returnsInstance() {
        val stitches = GaugeCount(12)
        val rows = GaugeCount(14)
        val length = GaugeSize(10.0)
        val gauge = Gauge(stitches, rows, length)

        assertEquals(stitches, gauge.nrOfStitches)
        assertEquals(rows, gauge.nrOfRows)
        assertEquals(length, gauge.size)
    }

    @Test
    fun calculateHeightFor_validGauge_returnsCorrectHeight(){
        val stitches = GaugeCount(12)
        val rows = GaugeCount(20)
        val length = GaugeSize(10.0)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateHeightFor(Count(13))

        assertEquals(Size(6.5), result)
    }

    @Test
    fun calculateWidthFor_validGauge_returnsCorrectWidth(){
        val stitches = GaugeCount(12)
        val rows = GaugeCount(20)
        val length = GaugeSize(10.0)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateWidthFor(Count(24))

        assertEquals(Size(20.0), result)
    }

    @Test
    fun calculateWidthFor_calculationYieldsDecimalResult_returnsValueRoundedToOneDecimalPlace() {
        val stitches = GaugeCount(11)
        val rows = GaugeCount(20)
        val length = GaugeSize(1.5)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateWidthFor(Count(10))
        assertEquals(1.4, result.value, 1e-3)
    }

    @Test
    fun calculateHeightFor_calculationYieldsDecimalResult_returnsValueRoundedToOneDecimalPlace() {
        val stitches = GaugeCount(11)
        val rows = GaugeCount(11)
        val length = GaugeSize(1.5)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateHeightFor(Count(10))
        assertEquals(1.4, result.value, 1e-3)
    }

    @Theory
    fun calculateStitchesFor_calculationYieldsDecimalResult_returnsValueRoundedToTheNextPositiveIntegerValue(testCase: SizeTestCase) {
        val stitches = GaugeCount(12)
        val rows = GaugeCount(20)
        val length = GaugeSize(10.0)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateStitchesFor(testCase.size)!!
        assertEquals(testCase.expected, result)
    }

    @Test
    fun calculateStitchesFor_calculationYieldsLongOverflow_returnsNull() {
        val stitches = GaugeCount(99)
        val rows = GaugeCount(20)
        val length = GaugeSize(1.0)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateStitchesFor(Size(Long.MAX_VALUE.toDouble()))
        assertNull(result)
    }

    @Theory
    fun calculateRowsFor_calculationYieldsDecimalResult_returnsValueRoundedToTheNextPositiveIntegerValue(testCase: SizeTestCase) {
        val stitches = GaugeCount(20)
        val rows = GaugeCount(12)
        val length = GaugeSize(10.0)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateRowsFor(testCase.size)!!
        assertEquals(testCase.expected, result)
    }

    @Test
    fun calculateRowsFor_calculationYieldsLongOverflow_returnsNull() {
        val stitches = GaugeCount(99)
        val rows = GaugeCount(20)
        val length = GaugeSize(1.0)
        val gauge = Gauge(stitches, rows, length)

        val result = gauge.calculateRowsFor(Size(Long.MAX_VALUE.toDouble()))
        assertNull(result)
    }

    companion object {
        @JvmField
        @DataPoints
        val testCases = arrayOf(
            SizeTestCase( Size(120.0), Count(144)),
            SizeTestCase( Size(123.0), Count(148)),
            SizeTestCase( Size(121.0), Count(146))
        )
    }

    data class SizeTestCase(
        val size: Size,
        val expected: Count
    )
}