package net.damschen.swatchit.test.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.resultWrappers.ValidationResult
import org.junit.Assert.assertEquals
import org.junit.Assert.assertThrows
import org.junit.Assert.assertTrue
import org.junit.Test

class GaugeSizeTests {

    @Test
    fun init_validInput_createsGaugeSize() {
        val value = 13.2
        val gaugeSize = GaugeSize(value)
        assertEquals(value, gaugeSize.value, 1e-2)
    }

    @Test
    fun init_negativeNumber_throws() {
        assertThrows(IllegalArgumentException::class.java) {
            GaugeSize(-1.3)
        }
    }

    @Test
    fun init_numberGreaterThan99_throws() {
        assertThrows(IllegalArgumentException::class.java) {
            GaugeSize(100.0)
        }
    }

    @Test
    fun validate_validInput_returnsSuccess() {
        val validationResult = GaugeSize.validate("13.2")
        assertTrue(validationResult is ValidationResult.Success)
    }

    @Test
    fun validate_validInputWithTrailingWhitespace_returnsSuccess() {
        val validationResult = GaugeSize.validate("  13.2   ")
        assertTrue(validationResult is ValidationResult.Success)
    }

    @Test
    fun validate_negativeNumber_returnsErrorMessageId() {
        val validationResult = GaugeSize.validate("-1")
        assertEquals(
            R.string.must_be_greater_than_zero,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_zero_returnsErrorMessageId() {
        val validationResult = GaugeSize.validate("0")
        assertEquals(
            R.string.must_be_greater_than_zero,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_numberGreaterThan99_returnErrorMessageId() {
        val validationResult = GaugeSize.validate("100")
        assertEquals(R.string.max_99, (validationResult as ValidationResult.Error).errorMessageId)
    }

    @Test
    fun validate_twoDigitsAfterDecimal_returnErrorMessageId() {
        val validationResult = GaugeSize.validate("13.24")
        assertEquals(
            R.string.max_1_decimal,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_valueNaN_returnErrorMessageId() {
        val validationResult = GaugeSize.validate("abc")
        assertEquals(
            R.string.invalid_number,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_valueEmpty_returnErrorMessageId() {
        val validationResult = GaugeSize.validate(" ")
        assertEquals(
            R.string.mandatory,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun create_validInput_returnsInstance() {
        val size = GaugeSize.create("13.2")
        assertEquals(13.2, size.value, 1e-2)
    }

    @Test
    fun create_valueNaN_throws() {
        assertThrows(IllegalArgumentException::class.java) { GaugeSize.create("abc") }
    }

    @Test
    fun create_inputWithTrailingWhitespace_returnsInstance() {
        val size = GaugeSize.create("  13  ")
        assertEquals(13.0, size.value, 1e-2)
    }

    @Test
    fun toString_validValue_printsValueWithoutTrailingZeros() {
        val gauge = GaugeSize(50.0)
        assertEquals("50", gauge.toString())
    }
}