package net.damschen.swatchit.test.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.aggregates.swatch.Count
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.domain.resultWrappers.ValidationResult
import org.junit.Assert.assertEquals
import org.junit.Assert.assertNull
import org.junit.Assert.assertThrows
import org.junit.Assert.assertTrue
import org.junit.Test

class CountTests {
    @Test
    fun init_validInput_createsCount() {
        val count = Count(13)
        assertEquals(13, count.value)
    }

    @Test
    fun init_negativeNumber_throws() {
        assertThrows(IllegalArgumentException::class.java) {
            Count(-1)
        }
    }

    @Test
    fun validate_validInput_returnsSuccess() {
        val validationResult = Count.validate("13")
        assertTrue(validationResult is ValidationResult.Success)
    }

    @Test
    fun validate_validInputWithTrailingWhitespace_returnsSuccess() {
        val validationResult = Count.validate(" 13   ")
        assertTrue(validationResult is ValidationResult.Success)
    }

    @Test
    fun validate_negativeNumber_returnsErrorMessageId() {
        val validationResult = Count.validate("-1")
        assertEquals(
            R.string.must_be_greater_than_zero,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_zero_returnsErrorMessageId() {
        val validationResult = Count.validate("0")
        assertEquals(
            R.string.must_be_greater_than_zero,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_valueNaN_returnsErrorMessageId() {
        val validationResult = Count.validate("abc")
        assertEquals(
            R.string.invalid_number,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun validate_valueEmpty_returnsSuccess() {
        val validationResult = Count.validate("")
        assertTrue(validationResult is ValidationResult.Success)
    }

    @Test
    fun validate_greaterThanMaxValue_returnErrorMessageId() {
        val validationResult =
            Count.validate((Long.MAX_VALUE.toDouble() + 1).toPlainStringWithoutTrailingZeros())
        assertEquals(
            R.string.number_too_big,
            (validationResult as ValidationResult.Error).errorMessageId
        )
    }

    @Test
    fun create_validInput_returnsInstance() {
        val count = Count.create("13")
        assertEquals(13, count!!.value)
    }

    @Test
    fun create_inputWithTrailingWhitespace_returnsInstance() {
        val count = Count.create(" 13  ")
        assertEquals(13, count!!.value)
    }

    @Test
    fun create_valueNaN_throws() {
        assertThrows(IllegalArgumentException::class.java) { Count.create("abc") }
    }

    @Test
    fun create_valueEmpty_returnsNull() {
        assertNull(Count.create(""))
    }

    @Test
    fun toString_validValue_printsValue() {
        val count = Count(12)
        assertEquals(12.toString(), count.toString())
    }
}