package net.damschen.swatchit.integrationTest.ui.screens

import androidx.compose.ui.test.assertCountEquals
import androidx.compose.ui.test.assertTextContains
import androidx.compose.ui.test.isDisplayed
import androidx.compose.ui.test.isNotDisplayed
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onChildAt
import androidx.compose.ui.test.onChildren
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.onNodeWithText
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performTextInput
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import dagger.hilt.android.testing.HiltTestApplication
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.MeasurementType
import net.damschen.swatchit.domain.aggregates.swatch.Name
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.infrastructure.database.AppDatabase
import net.damschen.swatchit.infrastructure.database.KnittingNeedleSize
import net.damschen.swatchit.infrastructure.database.MeasurementDao
import net.damschen.swatchit.infrastructure.database.MeasurementEntity
import net.damschen.swatchit.infrastructure.database.SwatchDao
import net.damschen.swatchit.infrastructure.database.SwatchEntity
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.ui.enums.CountType
import net.damschen.swatchit.ui.screens.MeasurementsScreen
import net.damschen.swatchit.ui.viewmodels.MeasurementViewModel
import org.junit.After
import org.junit.Assert.assertEquals
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.annotation.Config
import org.robolectric.shadows.ShadowLog
import java.time.LocalDate
import java.time.ZoneId
import javax.inject.Inject
import javax.inject.Named

@RunWith(RobolectricTestRunner::class)
@HiltAndroidTest
@Config(application = HiltTestApplication::class)
class MeasurementsScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    @get:Rule
    var hiltRule = HiltAndroidRule(this)

    @Inject
    @Named("test_db")
    lateinit var database: AppDatabase

    @Inject
    lateinit var swatchDao: SwatchDao

    @Inject
    lateinit var measurementDao: MeasurementDao

    @Inject
    lateinit var repository: SwatchRepository

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initDb() {
        hiltRule.inject()
        database.clearAllTables()
    }

    @After
    fun closeDb() {
        database.close()
    }

    @Test
    fun measurementInDatabase_showsMeasurement() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatchEntity).toInt()
        measurementDao.insert(createMeasurementEntity(id))
        composeTestRule.setContent {
            MeasurementsScreen(
                id, MeasurementViewModel(id, repository)
            ) {}
        }
        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)

        composeTestRule.onNodeWithTag("measurement_0_count")
            .assertTextContains(defaultMeasurement.gaugeCount.value.toString(), substring = true)
        composeTestRule.onNodeWithTag("measurement_0_count")
            .assertTextContains(defaultMeasurement.measurementType.name, substring = true)
        composeTestRule.onNodeWithTag("measurement_0_size")
            .assertTextContains(defaultMeasurement.size.value.toString(), substring = true)
    }

    @Test
    fun addButtonInAddDialogClicked_closesDialogAndUpdatesList() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatchEntity).toInt()
        measurementDao.insert(createMeasurementEntity(id))
        composeTestRule.setContent {
            MeasurementsScreen(
                id, MeasurementViewModel(id, repository)
            ) {}
        }
        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("Count").performTextInput("25")
        composeTestRule.onNodeWithTag("Size").performTextInput("17")
        composeTestRule.onNodeWithTag("CountType").performClick()
        composeTestRule.onNodeWithText(CountType.Rows.name).performClick()

        composeTestRule.onNodeWithTag("AddMeasurementButton").performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("AddMeasurementDialog").isNotDisplayed()
        }

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("measurement_1_count").isDisplayed()
        }

        composeTestRule.onNodeWithTag("measurement_1_count")
            .assertTextContains("25", substring = true)
        composeTestRule.onNodeWithTag("measurement_1_count")
            .assertTextContains(CountType.Rows.name, substring = true)
        composeTestRule.onNodeWithTag("measurement_1_size")
            .assertTextContains("17", substring = true)
    }

    @Test
    fun deleteClicked_deletesMeasurementFromDatabase() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatchEntity).toInt()
        measurementDao.insert(createMeasurementEntity(id))
        composeTestRule.setContent {
            MeasurementsScreen(
                id, MeasurementViewModel(id, repository)
            ) {}
        }
        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)

        composeTestRule.onNodeWithTag("optionsMenu_0").performClick()
        composeTestRule.onNodeWithTag("DeleteButton").performClick()
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("measurement_0_count").isNotDisplayed()
        }

        val measurements = measurementDao.getBySwatchId(id)
        assertEquals(0, measurements.size)
    }

    @Test
    fun deleteClicked_deletesMeasurementFromList() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatchEntity).toInt()
        measurementDao.insert(createMeasurementEntity(id))
        composeTestRule.setContent {
            MeasurementsScreen(
                id, MeasurementViewModel(id, repository)
            ) {}
        }
        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)

        composeTestRule.onNodeWithTag("optionsMenu_0").performClick()
        composeTestRule.onNodeWithTag("DeleteButton").performClick()
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("Column").onChildAt(0).isNotDisplayed()
        }
    }
}

val defaultMeasurement =
    Measurement(GaugeCount(3), GaugeSize(12.0), MeasurementType.Stitches)
private val defaultPattern = Pattern.create("Stockinette")!!
private val defaultGauge = Gauge(
    nrOfStitches = GaugeCount(20), nrOfRows = GaugeCount(30), size = GaugeSize(10.0)
)
private val defaultYarn = Yarn.create(Name.create("Yarn Name"), Name.create("Yarn Manufacturer"))!!
private val defaultNotes = Notes.create("Test notes!")!!
private val defaultCreatedAt = EpochMillis(
    LocalDate.of(2025, 2, 17).atStartOfDay(
        ZoneId.of("UTC")
    ).toInstant().toEpochMilli()
)
private val defaultName = Name.create("TestName")!!

private val defaultSwatchEntity = SwatchEntity(
    needleSize = KnittingNeedleSize.SIZE_2_5,
    yarnName = defaultYarn.name!!.value,
    yarnManufacturer = defaultYarn.manufacturer!!.value,
    nrOfStitches = defaultGauge.nrOfStitches.value,
    nrOfRows = defaultGauge.nrOfRows.value,
    gaugeLength = defaultGauge.size.value,
    createdAt = defaultCreatedAt.value,
    name = defaultName.value,
    pattern = defaultPattern.value,
    notes = defaultNotes.value,
    photoUUID = FakeUUIDProvider.defaultUUID
)

private fun createMeasurementEntity(swatchId: Int): MeasurementEntity {
    return MeasurementEntity(
        net.damschen.swatchit.infrastructure.database.MeasurementType.Stitches,
        defaultMeasurement.gaugeCount.value,
        defaultMeasurement.size.value,
        swatchId
    )
}