package net.damschen.swatchit.testHelpers

import android.content.Context
import androidx.room.Room
import dagger.Module
import dagger.Provides
import dagger.hilt.android.qualifiers.ApplicationContext
import dagger.hilt.components.SingletonComponent
import dagger.hilt.testing.TestInstallIn
import net.damschen.swatchit.infrastructure.database.AppDatabase
import net.damschen.swatchit.infrastructure.database.MeasurementDao
import net.damschen.swatchit.infrastructure.database.RoomTransactionProvider
import net.damschen.swatchit.infrastructure.database.SwatchDao
import net.damschen.swatchit.infrastructure.database.TransactionProvider
import net.damschen.swatchit.infrastructure.services.BitmapDecoder
import net.damschen.swatchit.di.DatabaseModule
import net.damschen.swatchit.domain.providers.UUIDProvider
import net.damschen.swatchit.shared.testhelpers.FakeBitmapDecoder
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.ui.providers.DateTimeProvider
import javax.inject.Named
import javax.inject.Singleton

@TestInstallIn(
    components = [SingletonComponent::class], replaces = [DatabaseModule::class]
)
@Module
object TestAppModule {
    @Provides
    @Singleton
    @Named("test_db")
    fun provideInMemoryDb(@ApplicationContext context: Context) = Room.inMemoryDatabaseBuilder(
        context, AppDatabase::class.java
    ).allowMainThreadQueries().build()

    @Provides
    @Singleton
    fun provideSwatchDao(@Named("test_db") database: AppDatabase): SwatchDao {
        return database.swatchDao()
    }

    @Provides
    @Singleton
    fun provideMeasurementDao(@Named("test_db") database: AppDatabase): MeasurementDao {
        return database.measurementDao()
    }

    @Provides
    @Singleton
    fun provideTransactionProvider(@Named("test_db") database: AppDatabase): TransactionProvider {
        return RoomTransactionProvider(database)
    }
}

@TestInstallIn(
    components = [SingletonComponent::class],
    replaces = [net.damschen.swatchit.di.DateTimeProviderModule::class]
)
@Module
object DateTimeProviderModule {
    @Provides
    @Singleton
    fun provideDateTimeProvider(): DateTimeProvider = FakeDateTimeProvider()
}

@TestInstallIn(
    components = [SingletonComponent::class],
    replaces = [net.damschen.swatchit.di.UUIDProviderModule::class]
)
@Module
object UUIDProviderModule {
    @Provides
    @Singleton
    fun provideUUIDProvider(): UUIDProvider = FakeUUIDProvider()
}

@TestInstallIn(
    components = [SingletonComponent::class],
    replaces = [net.damschen.swatchit.di.BitmapDecoderModule::class]
)
@Module
object BitmapDecoderModule {
    @Provides
    @Singleton
    fun provideBitmapDecoder(): BitmapDecoder = FakeBitmapDecoder()
}