package net.damschen.swatchit.integrationTest

import android.text.format.DateFormat
import androidx.compose.ui.test.assert
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.assertTextContains
import androidx.compose.ui.test.assertTextEquals
import androidx.compose.ui.test.hasText
import androidx.compose.ui.test.isDisplayed
import androidx.compose.ui.test.isEnabled
import androidx.compose.ui.test.isNotDisplayed
import androidx.compose.ui.test.junit4.createAndroidComposeRule
import androidx.compose.ui.test.onChildren
import androidx.compose.ui.test.onNodeWithContentDescription
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.onNodeWithText
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.compose.ui.test.performTextInput
import androidx.compose.ui.test.performTextReplacement
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import dagger.hilt.android.testing.HiltTestApplication
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.MainActivity
import net.damschen.swatchit.infrastructure.database.AppDatabase
import net.damschen.swatchit.infrastructure.database.MeasurementDao
import net.damschen.swatchit.infrastructure.database.MeasurementEntity
import net.damschen.swatchit.infrastructure.database.MeasurementType
import net.damschen.swatchit.infrastructure.database.SwatchAggregate
import net.damschen.swatchit.infrastructure.database.SwatchDao
import net.damschen.swatchit.infrastructure.database.SwatchEntity
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import org.junit.After
import org.junit.Assert.assertEquals
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.annotation.Config
import org.robolectric.shadows.ShadowLog
import java.time.LocalDate
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import javax.inject.Inject
import javax.inject.Named

@RunWith(RobolectricTestRunner::class)
@HiltAndroidTest
@Config(application = HiltTestApplication::class)
class MainActivityTests {

    @get:Rule(order = 0)
    var hiltRule = HiltAndroidRule(this)

    @get:Rule(order = 1)
    val composeTestRule = createAndroidComposeRule<MainActivity>()

    @Inject
    @Named("test_db")
    lateinit var database: AppDatabase

    @Inject
    lateinit var swatchDao: SwatchDao

    @Inject
    lateinit var measurementDao: MeasurementDao


    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun init() {
        hiltRule.inject()
        database.clearAllTables()
    }

    @After
    fun closeDb() {
        database.close()
    }

    @Test
    fun addSwatch_validInput_addsSwatchToDatabase() = runTest {
        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("Name").performScrollTo().performTextInput(defaultSwatch.name)
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextInput(defaultSwatch.pattern)
        composeTestRule.onNodeWithTag("YarnManufacturer").performScrollTo()
            .performTextInput(defaultSwatch.yarnManufacturer)
        composeTestRule.onNodeWithTag("YarnName").performScrollTo()
            .performTextInput(defaultSwatch.yarnName)
        composeTestRule.onNodeWithTag("NeedleSize").performScrollTo().performClick()
        composeTestRule.onNodeWithText(defaultNeedleSize.displayValue).performScrollTo()
            .performClick()
        composeTestRule.onNodeWithTag("Notes").performScrollTo()
            .performTextInput(defaultSwatch.notes)
        composeTestRule.onNodeWithTag("CreatedAt").performScrollTo().performClick()
        // Wait for the date picker dialog to appear
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("DatePickerDialog").isDisplayed()
        }
        composeTestRule.onNode(hasText("Selected date")).performClick()
        composeTestRule.onNodeWithContentDescription("Switch to text input mode").performClick()
        composeTestRule.onNodeWithContentDescription("Date, MM/DD/YYYY").performTextInput(
            defaultLocalDate.format(
                DateTimeFormatter.ofPattern("MMddYYYY")
            )
        )
        composeTestRule.onNodeWithTag("DatePickerOkButton").performClick()

        composeTestRule.onNodeWithTag(addSwatchButtonTag).performClick()

        val swatchesInDb = swatchDao.get().first()
        val swatchAggregate = swatchesInDb.first()
        val expected = SwatchAggregate(
            defaultSwatch.copy(
                gaugeLength = null, nrOfStitches = null, nrOfRows = null, photoUUID = null
            ), emptyList()
        )
        assertEquals(expected, swatchAggregate)
    }

    @Test
    fun addSwatch_minimalValidInput_addsSwatchToDatabase() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag(addSwatchButtonTag).performClick()

        val swatchesInDb = swatchDao.get().first()
        val swatchAggregate = swatchesInDb.first()
        assertEquals(minimalSwatchAggregate, swatchAggregate)
    }

    @Test
    fun deleteSwatch_validInput_removesSwatchFromDb() = runTest {
        swatchDao.insert(swatch = defaultSwatch)
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }

        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag("DeleteButton").performClick()
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        val swatchesInDb = swatchDao.get().first()
        assertEquals(0, swatchesInDb.count())
    }

    @Test
    fun navigation_validInput_navigatesToMeasurementsView() = runTest {
        swatchDao.insert(swatch = defaultSwatch)
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }

        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag("ViewMeasurementsDropdownItem").performClick()

        composeTestRule.onNodeWithTag("MeasurementsTitle").assertExists()
    }

    @Test
    fun navigation_validInput_navigatesToCalculationsView() = runTest {
        swatchDao.insert(swatch = defaultSwatch)
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }

        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag("CalculationsDropdownItem").performClick()

        composeTestRule.onNodeWithTag("CalculationsTitle").assertExists()
    }

    @Test
    fun editSwatch_nothingHasChanged_saveChangesButtonIsDisabled() = runTest {
        swatchDao.insert(swatch = defaultSwatch).toInt()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }

        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        composeTestRule.onNodeWithTag(saveChangesButtonTag).assert(!isEnabled())
    }

    @Test
    fun swatchDetails_gaugeDialogNoValuesChanged_saveButtonIsNotEnabled() = runTest {
        swatchDao.insert(swatch = defaultSwatch).toInt()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag("SaveGaugeButton").assertIsNotEnabled()
    }

    @Test
    fun swatchDetails_gaugeDialogSaved_measurementsStillInDatabase() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatch).toInt()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(viewMeasurementsDropDownItemTag).performClick()

        // MeasurementsScreen
        composeTestRule.onNodeWithTag("AddButton").performClick()

        composeTestRule.onNodeWithTag("Count").performTextInput("25")
        composeTestRule.onNodeWithTag("Size").performTextInput("17")

        composeTestRule.onNodeWithTag("AddMeasurementButton").performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("measurement_0_count").isDisplayed()
        }

        composeTestRule.onNodeWithTag("BackButton").performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag("GaugeLengthInput").performTextReplacement("25")

        composeTestRule.onNodeWithTag("SaveGaugeButton").performClick()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("EditGaugeDialog").isNotDisplayed()
        }
        assertEquals(1, measurementDao.getBySwatchId(id).size)
    }

    @Test
    fun editSwatch_validInput_savesToDb() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatch).toInt()
        val measurementEntity = createMeasurementEntity(id)
        measurementDao.insert(measurement = measurementEntity)
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }

        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        // EditSwatchScreen
        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)

        composeTestRule.onNodeWithTag(saveChangesButtonTag).performClick()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag(editGaugeButtonTag).isDisplayed()
        }

        val swatchesInDb = swatchDao.get().first()
        val swatchAggregate = swatchesInDb.first()
        val expected =
            SwatchAggregate(defaultSwatch.copy(pattern = newPattern), listOf(measurementEntity))
        assertEquals(expected, swatchAggregate)
    }

    @Test
    fun editSwatch_editPatternAndGauge_savesAllChangesToDb() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatch).toInt()
        val measurementEntity = createMeasurementEntity(id)
        measurementDao.insert(measurement = measurementEntity)
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }

        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        // EditSwatchScreen
        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)


        composeTestRule.onNodeWithTag(saveChangesButtonTag).performClick()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag(editGaugeButtonTag).isDisplayed()
        }

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag("NrOfStitchesInput").performScrollTo()
            .performTextReplacement("14")

        composeTestRule.onNodeWithTag("SaveGaugeButton").performClick()

        val swatchesInDb = swatchDao.get().first()
        val swatchAggregate = swatchesInDb.first()
        val expected = SwatchAggregate(
            defaultSwatch.copy(pattern = newPattern, nrOfStitches = 14), listOf(measurementEntity)
        )
        composeTestRule.waitForIdle()
        assertEquals(expected, swatchAggregate)
    }

    @Test
    fun addSwatch_nameChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("Name").performScrollTo().performTextInput(defaultSwatch.name)


        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag("Name").assert(hasText(""))
    }

    @Test
    fun addSwatch_patternChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextInput(defaultSwatch.pattern)

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag(patternInputTag).assert(hasText(""))
    }

    @Test
    fun addSwatch_yarnManufacturerChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("YarnManufacturer").performScrollTo()
            .performTextInput(defaultSwatch.yarnManufacturer)

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag("YarnManufacturer").assert(hasText(""))
    }

    @Test
    fun addSwatch_yarnNameChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("YarnName").performScrollTo()
            .performTextInput(defaultSwatch.yarnName)

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag("YarnName").assert(hasText(""))
    }

    @Test
    fun addSwatch_needleSizeChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("NeedleSize").performScrollTo().performClick()

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag("NeedleSize")
            .assert(hasText(KnittingNeedleSize.SIZE_2_5.displayValue))
    }

    @Test
    fun addSwatch_notesChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("Notes").performScrollTo()
            .performTextInput(defaultSwatch.notes)

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag("Notes").assert(hasText(""))
    }

    @Test
    fun addSwatch_createdAtChangedAndBackButtonClicked_resetsState() = runTest {

        composeTestRule.onNodeWithTag(addButtonTag).performClick()
        composeTestRule.onNodeWithTag("CreatedAt").performScrollTo().performClick()
        // Wait for the date picker dialog to appear
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("DatePickerDialog").isDisplayed()
        }
        composeTestRule.onNode(hasText("Selected date")).performClick()
        composeTestRule.onNodeWithContentDescription("Switch to text input mode").performClick()
        composeTestRule.onNodeWithContentDescription("Date, MM/DD/YYYY").performTextInput(
            defaultLocalDate.format(
                DateTimeFormatter.ofPattern("MMddYYYY")
            )
        )
        composeTestRule.onNodeWithTag("DatePickerOkButton").performClick()

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        val formattedDate =
            DateFormat.format("dd/MM/yyyy", FakeDateTimeProvider.date)
                .toString()
        composeTestRule.onNodeWithTag("CreatedAt").assert(hasText(formattedDate))
    }

    @Test
    fun editSwatch_patternChangedAndBackButtonClicked_showsOldPatternInDetailsView() = runTest {
        swatchDao.insert(swatch = defaultSwatch).toInt()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }
        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        // EditSwatchScreen
        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)

        composeTestRule.onNodeWithTag(backButtonTag).performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(patternInputTag).assertTextEquals(defaultSwatch.pattern)
    }

    @Test
    fun editSwatch_patternChangedAndBackButtonClicked_resetsState() = runTest {
        swatchDao.insert(swatch = defaultSwatch).toInt()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }
        // SwatchListScreen
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        // EditSwatchScreen
        val newPattern = "New Test Pattern"
        composeTestRule.onNodeWithTag(patternInputTag).performScrollTo()
            .performTextReplacement(newPattern)


        composeTestRule.onNodeWithTag(backButtonTag).performClick()

        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        composeTestRule.onNodeWithTag(patternInputTag).assert(hasText(defaultSwatch.pattern))
    }

    @Test
    fun measurements_calculateAndSaveGauge_GaugeDisplayedInDetailsView() = runTest {
        val id = swatchDao.insert(swatch = defaultSwatch).toInt()
        val measurementEntity = MeasurementEntity(
            MeasurementType.Stitches,
            40,
            20.0,
            id
        )
        measurementDao.insert(
            measurements = listOf(
                measurementEntity,
                measurementEntity.copy(type = MeasurementType.Rows)
            )
        )

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithText(defaultSwatch.name).isDisplayed()
        }
        val columnItems = composeTestRule.onNodeWithTag(swatchesColumnTag).onChildren()
        columnItems[0].performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(viewMeasurementsDropDownItemTag).performClick()

        // MeasurementsScreen
        composeTestRule.onNodeWithTag("CalculateGaugeButton").performClick()
        composeTestRule.onNodeWithTag("SaveGaugeCalculation").performClick()
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("CalculateGaugeDialog").isNotDisplayed()
        }

        composeTestRule.onNodeWithTag("BackButton").performClick()

        // SwatchDetailsScreen
        composeTestRule.onNodeWithTag("NrOfStitches").assertTextContains("20")
        composeTestRule.onNodeWithTag("NrOfRows").assertTextContains("20")
        composeTestRule.onNodeWithTag("GaugeLength").assertTextContains("10 cm")
    }
}

private const val addButtonTag = "AddButton"
private const val addSwatchButtonTag = "AddSwatchButton"
private const val swatchesColumnTag = "Column"
private const val optionsMenuTag = "OptionsMenu"
private const val editButtonTag = "EditButton"
private const val editGaugeButtonTag = "EditGaugeButton"
private const val saveChangesButtonTag = "SaveChangesButton"
private const val patternInputTag = "Pattern"
private const val cancelButtonTag = "CancelButton"
private const val backButtonTag = "BackButton"
private const val viewMeasurementsDropDownItemTag = "ViewMeasurementsDropdownItem"

private val defaultLocalDate: LocalDate = LocalDate.of(2015, 2, 17)

private val defaultUtcDate: Long =
    defaultLocalDate.atStartOfDay(ZoneId.of("UTC")).toInstant().toEpochMilli()
private val defaultNeedleSize: KnittingNeedleSize = KnittingNeedleSize.SIZE_0_5
private val defaultSwatch: SwatchEntity = SwatchEntity(
    needleSize = net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_0_5,
    yarnName = "Yarn Name",
    yarnManufacturer = "Manufacturer",
    nrOfStitches = 13,
    nrOfRows = 21,
    gaugeLength = 32.0,
    createdAt = defaultUtcDate,
    name = "TestName",
    pattern = "TestPattern",
    notes = "Test Notes!",
    photoUUID = FakeUUIDProvider.defaultUUID
)


private val minimalSwatchAggregate = SwatchAggregate(
    SwatchEntity(
        needleSize = net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_5,
        yarnName = "",
        yarnManufacturer = "",
        nrOfStitches = null,
        nrOfRows = null,
        gaugeLength = null,
        createdAt = FakeDateTimeProvider.date.time,
        name = "",
        pattern = "",
        notes = "",
        photoUUID = null
    ), emptyList()
)

private fun createMeasurementEntity(swatchId: Int): MeasurementEntity {
    return MeasurementEntity(
        MeasurementType.Stitches,
        34,
        13.0,
        swatchId
    )
}