package net.damschen.swatchit.ui.screens

import androidx.activity.compose.BackHandler
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.text.TextAutoSize
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material.icons.filled.Calculate
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.MoreVert
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuAnchorType
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.FabPosition
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.compose.ui.window.Dialog
import androidx.hilt.lifecycle.viewmodel.compose.hiltViewModel
import net.damschen.swatchit.R
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.ui.elements.ConfirmDeleteDialog
import net.damschen.swatchit.ui.elements.ErrorMessage
import net.damschen.swatchit.ui.elements.ProgressIndicator
import net.damschen.swatchit.ui.elements.ValidatedTextField
import net.damschen.swatchit.ui.enums.CountType
import net.damschen.swatchit.ui.models.GaugeCalculationState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.MeasurementFormState
import net.damschen.swatchit.ui.models.MeasurementListItem
import net.damschen.swatchit.ui.viewmodels.MeasurementViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun MeasurementsScreen(
    itemId: Int, viewModel: MeasurementViewModel = hiltViewModel(
        creationCallback = { factory: MeasurementViewModel.Factory ->
            factory.create(itemId)
        }), onNavigateBack: () -> Unit
) {
    val measurementsState by viewModel.measurementsState.collectAsState()
    val measurementFormState by viewModel.measurementFormState.collectAsState()
    val gaugeCalculationState by viewModel.gaugeCalculationState.collectAsState()
    val loadState by viewModel.loadState.collectAsState()
    var addDialogVisible by remember { mutableStateOf(false) }
    var calculateDialogVisible by remember { mutableStateOf(false) }
    val addSuccessful by viewModel.savedSuccessfully.collectAsState(true)
    val gaugeUpdateSuccessful by viewModel.gaugeSuccessfullySaved.collectAsState(true)
    val snackbarHostState = remember { SnackbarHostState() }

    LaunchedEffect(Unit) {
        viewModel.savedSuccessfully.collect { success ->
            if (success) {
                addDialogVisible = false
                viewModel.initFormState()
            }
        }
    }

    LaunchedEffect(Unit) {
        viewModel.gaugeSuccessfullySaved.collect { success ->
            if (success) {
                calculateDialogVisible = false
            }
        }
    }

    val errorText = stringResource(R.string.error_while_deleting_measurement)
    LaunchedEffect(Unit) {
        viewModel.deletedSuccessfully.collect { success ->
            if (!success) {
                snackbarHostState.showSnackbar(
                    message = errorText, duration = SnackbarDuration.Short
                )
            }
        }
    }

    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())

    BackHandler { onNavigateBack() }

    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        snackbarHost = {
            SnackbarHost(hostState = snackbarHostState)
        },
        topBar = {
            CenterAlignedTopAppBar(
                title = {
                    Text(
                        stringResource(R.string.measurements),
                        maxLines = 1,
                        overflow = TextOverflow.Ellipsis,
                        modifier = Modifier.testTag("MeasurementsTitle")
                    )
                },
                actions = {
                    IconButton(
                        modifier = Modifier.testTag("CalculateGaugeButton"),
                        onClick = { calculateDialogVisible = true }
                    ) {
                        Icon(
                            imageVector = Icons.Filled.Calculate,
                            contentDescription = stringResource(R.string.edit_button_description)
                        )
                    }
                },

                navigationIcon = {
                    IconButton(
                        onClick = onNavigateBack, modifier = Modifier.testTag("BackButton")
                    ) {
                        Icon(
                            imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                            contentDescription = stringResource(R.string.back_button_description)
                        )
                    }
                }, scrollBehavior = scrollBehavior
            )
        },
        floatingActionButton = {
            FloatingActionButton(
                onClick = { addDialogVisible = true }, modifier = Modifier.testTag("AddButton")
            ) {
                Icon(Icons.Filled.Add, stringResource(R.string.add_measurement_description))
            }
        },
        floatingActionButtonPosition = FabPosition.Center
    ) { innerPadding ->
        when (val state = loadState) {
            LoadState.Loading -> {
                ProgressIndicator(innerPadding)
            }

            is LoadState.Error -> {
                ErrorMessage(
                    message = state.message ?: stringResource(R.string.unknown_error),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding
                )
            }

            LoadState.NotFound -> {
                ErrorMessage(
                    message = stringResource(R.string.swatch_not_found),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding
                )
            }

            LoadState.Success -> {
                MeasurementsColumn(
                    innerPadding, measurementsState.items, onDeleteMeasurement = { index ->
                        viewModel.deleteMeasurementAt(index)
                    })
                when {
                    addDialogVisible -> {
                        AddMeasurementDialog(
                            measurementFormState,
                            viewModel,
                            addSuccessful,
                            onCancel = {
                                addDialogVisible = false
                                viewModel.initFormState()
                            },
                            onAdd = {
                                viewModel.addMeasurement()
                            })
                    }

                    calculateDialogVisible -> {
                        viewModel.calculateGauge()
                        CalculateGaugeDialog(gaugeCalculationState, onCancel = {
                            calculateDialogVisible = false
                        }, gaugeUpdateSuccessful = gaugeUpdateSuccessful, onSave = {
                            viewModel.saveGaugeCalculationState()
                        })
                    }
                }
            }

            LoadState.Initial -> {}
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun AddMeasurementDialog(
    state: MeasurementFormState,
    viewModel: MeasurementViewModel,
    addSuccessful: Boolean,
    onCancel: () -> Unit,
    onAdd: () -> Unit
) {
    var countTypeExpanded by remember { mutableStateOf(false) }
    Dialog(onDismissRequest = { onCancel() }) {
        Card(
            modifier = Modifier
                .fillMaxWidth()
                .height(450.dp)
                .padding(16.dp)
                .verticalScroll(rememberScrollState())
                .testTag("AddMeasurementDialog"),
            shape = RoundedCornerShape(16.dp),
        ) {
            Column(
                modifier = Modifier.fillMaxSize(),
                verticalArrangement = Arrangement.SpaceEvenly,
                horizontalAlignment = Alignment.CenterHorizontally,
            ) {
                Text(
                    text = stringResource(R.string.add_measurement),
                    modifier = Modifier.padding(16.dp),
                    style = MaterialTheme.typography.headlineSmall
                )
                if (!addSuccessful) {
                    UpdateErrorCard()
                }

                ValidatedTextField(
                    validatedInput = state.count,
                    onValueChange = {
                        viewModel.onCountChanged(it)
                    },
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    labelText = stringResource(id = R.string.count),
                    testTag = "Count"
                )

                ExposedDropdownMenuBox(
                    expanded = countTypeExpanded,
                    onExpandedChange = { countTypeExpanded = !countTypeExpanded },
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(24.dp, 6.dp)
                ) {
                    OutlinedTextField(
                        value = state.type.name,
                        onValueChange = {},
                        label = { Text(stringResource(R.string.count_type)) },
                        readOnly = true,
                        trailingIcon = {
                            Icon(
                                Icons.Default.ArrowDropDown,
                                contentDescription = stringResource(R.string.drop_down_description)
                            )
                        },
                        modifier = Modifier
                            .fillMaxWidth()
                            .menuAnchor(
                                type = ExposedDropdownMenuAnchorType.PrimaryEditable, enabled = true
                            )
                            .testTag("CountType")
                    )
                    ExposedDropdownMenu(
                        expanded = countTypeExpanded,
                        onDismissRequest = { countTypeExpanded = false }) {
                        CountType.entries.forEach { size ->
                            DropdownMenuItem(text = { Text(size.name) }, onClick = {
                                viewModel.onCountTypeChanged(size)
                                countTypeExpanded = false
                            })
                        }
                    }
                }

                ValidatedTextField(
                    validatedInput = state.size,
                    onValueChange = {
                        viewModel.onSizeChanged(it)
                    },
                    labelText = stringResource(id = R.string.size_cm),
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    testTag = "Size"
                )
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.Center,
                ) {
                    TextButton(
                        onClick = { onCancel() },
                        modifier = Modifier
                            .padding(8.dp)
                            .testTag("CancelAddMeasurementButton"),
                    ) {
                        Text(stringResource(R.string.cancel))
                    }
                    TextButton(
                        enabled = state.isValid(), onClick = {
                            onAdd()
                        }, modifier = Modifier
                            .padding(8.dp)
                            .testTag("AddMeasurementButton")
                    ) {
                        Text(stringResource(R.string.add))
                    }
                }
            }
        }
    }
}

@Composable
private fun MeasurementsColumn(
    innerPadding: PaddingValues,
    measurements: List<MeasurementListItem>,
    onDeleteMeasurement: (Int) -> Unit
) {
    Column {
        LazyColumn(
            modifier = Modifier
                .padding(innerPadding)
                .testTag("Column"),
            contentPadding = PaddingValues(horizontal = 16.dp, vertical = 16.dp),
            verticalArrangement = Arrangement.spacedBy(4.dp)
        ) {
            itemsIndexed(measurements) { index, measurementListItem ->
                MeasurementItem(
                    measurementListItem,
                    index,
                    onDeleteConfirmed = { onDeleteMeasurement(index) })
            }
        }
    }
}

@Composable
fun MeasurementItem(measurement: MeasurementListItem, index: Int, onDeleteConfirmed: () -> Unit) {
    Column {
        ElevatedCard(
            elevation = CardDefaults.cardElevation(
                defaultElevation = 6.dp
            ), modifier = Modifier
                .fillMaxWidth()
                .padding(4.dp)
        ) {
            Row {
                Text(
                    text = "${measurement.count} ${measurement.countType.name}",
                    modifier = Modifier
                        .padding(16.dp, 16.dp, 8.dp, 16.dp)
                        .testTag("measurement_${index}_count")
                        .weight(3f),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )

                Text(
                    text = "\u2259",
                    modifier = Modifier
                        .padding(8.dp, 16.dp, 8.dp, 16.dp)
                        .weight(1f),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )

                Text(
                    text = "${measurement.size} cm",
                    modifier = Modifier
                        .padding(8.dp, 16.dp, 16.dp, 16.dp)
                        .testTag("measurement_${index}_size")
                        .weight(2f),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )

                OptionsMenu(onDeleteConfirmed = onDeleteConfirmed, index)
            }
        }
    }
}


@Composable
private fun OptionsMenu(onDeleteConfirmed: () -> Unit, index: Int) {
    var expanded by remember { mutableStateOf(false) }
    var alertDialogVisible by remember { mutableStateOf(false) }

    IconButton(
        onClick = { expanded = !expanded },
        modifier = Modifier
            .testTag("optionsMenu_${index}")
            .padding(0.dp, 8.dp, 16.dp, 8.dp)
    ) {
        Icon(
            imageVector = Icons.Filled.MoreVert,
            contentDescription = stringResource(R.string.main_menu)
        )
        DropdownMenu(expanded = expanded, onDismissRequest = { expanded = false }) {
            DropdownMenuItem(
                text = { Text(stringResource(R.string.delete)) },
                modifier = Modifier.testTag("DeleteButton"),
                onClick = {
                    alertDialogVisible = true
                },
                leadingIcon = {
                    Icon(
                        imageVector = Icons.Filled.Delete,
                        contentDescription = stringResource(R.string.delete_measurement_description)
                    )
                })
            when {
                alertDialogVisible -> {
                    val text =
                        stringResource(R.string.clicking_confirm_will_permanently_delete_the_measurement)
                    ConfirmDeleteDialog(text, onDismiss = {
                        alertDialogVisible = false
                        expanded = false
                    }, onConfirmation = {
                        alertDialogVisible = false
                        expanded = false
                        onDeleteConfirmed()
                    })
                }
            }
        }
    }
}

@Composable
private fun CalculateGaugeDialog(
    gaugeCalculationState: GaugeCalculationState,
    gaugeUpdateSuccessful: Boolean,
    onCancel: () -> Unit,
    onSave: () -> Unit
) {
    Dialog(onDismissRequest = { onCancel() }) {
        Card(
            modifier = Modifier
                .fillMaxWidth()
                .wrapContentHeight()
                .padding(16.dp)
                .verticalScroll(rememberScrollState())
                .testTag("CalculateGaugeDialog"),
            shape = RoundedCornerShape(16.dp),
        ) {
            Column(
                modifier = Modifier.fillMaxSize(),
                verticalArrangement = Arrangement.SpaceEvenly,
                horizontalAlignment = Alignment.CenterHorizontally,
            ) {
                Text(
                    text = stringResource(R.string.calculated_gauge),
                    modifier = Modifier.padding(16.dp),
                    style = MaterialTheme.typography.headlineMedium
                )
                if (!gaugeUpdateSuccessful) {
                    UpdateErrorCard()
                }
                when (gaugeCalculationState) {
                    is GaugeCalculationState.Invalid -> {
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 24.dp),
                            colors = CardDefaults.cardColors(
                                containerColor = MaterialTheme.colorScheme.secondaryContainer
                            )
                        ) {
                            Text(
                                text = stringResource(
                                    gaugeCalculationState.errorMessageId
                                        ?: R.string.could_not_calculate
                                ),
                                modifier = Modifier
                                    .padding(12.dp)
                                    .testTag("CalculateGaugeErrorMessage"),
                                style = MaterialTheme.typography.bodyMedium,
                                color = MaterialTheme.colorScheme.onSecondaryContainer
                            )
                        }
                    }

                    is GaugeCalculationState.Valid -> {
                        Text(
                            text = stringResource(R.string.gauge_calculation_sub_heading) + ':',
                            modifier = Modifier
                                .padding(16.dp, 8.dp),
                            style = MaterialTheme.typography.titleMedium
                        )
                        Text(
                            text =
                                stringResource(
                                    R.string.sts_x_rows_per_cm,
                                    gaugeCalculationState.gaugeState.nrOfStitches,
                                    gaugeCalculationState.gaugeState.nrOfRows,
                                    gaugeCalculationState.gaugeState.gaugeSize.toPlainStringWithoutTrailingZeros()
                                ),
                            textAlign = TextAlign.Center,
                            modifier = Modifier
                                .padding(16.dp, 16.dp)
                                .testTag("GaugeCalculationResult"),
                            style = MaterialTheme.typography.bodyLarge,
                            autoSize = TextAutoSize.StepBased(12.sp, 20.sp, 1.sp),
                            overflow = TextOverflow.Ellipsis
                        )

                        Text(
                            text = stringResource(
                                R.string.save_gauge_info,
                                stringResource(R.string.save_gauge)
                            ),
                            modifier = Modifier
                                .padding(16.dp, 16.dp),
                            style = MaterialTheme.typography.bodyMedium
                        )
                    }

                    GaugeCalculationState.NotCalculated -> {
                        Card(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(horizontal = 24.dp),
                            colors = CardDefaults.cardColors(
                                containerColor = MaterialTheme.colorScheme.secondaryContainer
                            )
                        ) {
                            Text(
                                text = stringResource(R.string.could_not_calculate),
                                modifier = Modifier
                                    .padding(12.dp)
                                    .testTag("NotCalculatedErrorMessage"),
                                style = MaterialTheme.typography.bodyMedium,
                                color = MaterialTheme.colorScheme.onSecondaryContainer
                            )
                        }
                    }
                }
            }

            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.Center,
            ) {
                TextButton(
                    onClick = { onCancel() },
                    modifier = Modifier
                        .padding(8.dp)
                        .testTag("CancelCalculateGaugeButton"),
                ) {
                    Text(stringResource(R.string.cancel))
                }
                TextButton(
                    enabled = gaugeCalculationState is GaugeCalculationState.Valid, onClick = {
                        onSave()
                    }, modifier = Modifier
                        .padding(8.dp)
                        .testTag("SaveGaugeCalculation")
                ) {
                    Text(stringResource(R.string.save_gauge))
                }
            }
        }
    }
}

@Composable
private fun UpdateErrorCard() {
    Card(
        modifier = Modifier
            .fillMaxWidth()
            .padding(horizontal = 24.dp),
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.errorContainer
        )
    ) {
        Text(
            text = stringResource(R.string.error_while_updating_swatch),
            modifier = Modifier
                .padding(12.dp)
                .testTag("Error"),
            style = MaterialTheme.typography.bodySmall,
            color = MaterialTheme.colorScheme.onErrorContainer
        )
    }
}
