package net.damschen.swatchit.ui.screens

import androidx.activity.compose.BackHandler
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import net.damschen.swatchit.R
import net.damschen.swatchit.ui.elements.ErrorDialog
import net.damschen.swatchit.ui.elements.ErrorMessage
import net.damschen.swatchit.ui.elements.ProgressIndicator
import net.damschen.swatchit.ui.elements.SwatchPropertiesColumn
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun EditSwatchScreen(
    viewModel: EditSwatchViewModel, onNavigateBack: () -> Unit
) {
    val swatchState by viewModel.formManager.swatchFormState.collectAsState()
    val hasChanged by viewModel.swatchHasChanged.collectAsState()
    val loadState by viewModel.loadState.collectAsState()
    var showUpdateErrorDialog by remember { mutableStateOf(false) }

    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())

    LaunchedEffect(Unit) {
        viewModel.swatchFormSuccessfullySaved.collect { success ->
            if (success) onNavigateBack()
            else showUpdateErrorDialog = true
        }
    }

    BackHandler { onNavigateBack() }

    Scaffold(modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection), topBar = {
        CenterAlignedTopAppBar(
            title = {
                Text(
                    stringResource(R.string.edit), maxLines = 1, overflow = TextOverflow.Ellipsis
                )
            }, navigationIcon = {
                IconButton(onClick = onNavigateBack, modifier = Modifier.testTag("BackButton")) {
                    Icon(
                        imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                        contentDescription = stringResource(R.string.back_button_description)
                    )
                }
            }, actions = {
                TextButton(
                    modifier = Modifier.testTag("SaveChangesButton"), onClick = {
                        viewModel.saveSwatchForm()
                    }, enabled = swatchState.isValid() && hasChanged
                ) { Text(stringResource(R.string.save_changes)) }
            }, scrollBehavior = scrollBehavior
        )
    }) { innerPadding ->
        when (val state = loadState) {
            LoadState.Loading -> {
                ProgressIndicator(innerPadding)
            }

            is LoadState.Error -> {
                ErrorMessage(
                    message = state.message ?: stringResource(R.string.unknown_error),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding
                )
            }

            LoadState.NotFound -> {
                ErrorMessage(
                    message = stringResource(R.string.swatch_not_found),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding
                )
            }

            LoadState.Success -> SwatchPropertiesColumn(
                innerPadding,
                viewModel.formManager,
                swatchState
            )

            LoadState.Initial -> {}
        }

        if (showUpdateErrorDialog) {
            ErrorDialog(
                text = stringResource(R.string.error_while_updating_swatch),
                testTag = "UpdateErrorDialog",
                onCancel = { showUpdateErrorDialog = false },
                onRetry = {
                    showUpdateErrorDialog = false
                    viewModel.saveSwatchForm()
                })

        }
    }
}

