package net.damschen.swatchit.infrastructure.services

import android.content.Context
import android.graphics.Bitmap
import android.net.Uri
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.infrastructure.resultWrappers.PhotoResult
import java.io.File

interface PhotoStorageService {
    companion object {
        const val IMAGES_DIR_NAME = "images"
    }

    suspend fun copyToLocalFileSystem(uri: Uri, photo: Photo): PhotoResult
    fun deleteFromLocalFileSystem(photo: Photo): PhotoResult
}

class AndroidPhotoStorageService(
    context: Context,
    private val bitmapDecoder: BitmapDecoder
) : PhotoStorageService {

    private val imagesDir = File(context.filesDir, PhotoStorageService.IMAGES_DIR_NAME)

    private companion object {
        const val MAX_IMAGE_SIZE = 1440
        const val JPEG_QUALITY = 85
    }

    override suspend fun copyToLocalFileSystem(uri: Uri, photo: Photo): PhotoResult = runCatching {
        ensureDirectoryExists()

        val destinationFile = File(imagesDir, photo.fileName)

        bitmapDecoder.decode(uri, MAX_IMAGE_SIZE).use { bitmap ->
            destinationFile.outputStream().use { output ->
                bitmap.compress(Bitmap.CompressFormat.JPEG, JPEG_QUALITY, output)
            }
        }
        PhotoResult.Success
    }.getOrElse {
        PhotoResult.Error(it.message)
    }

    override fun deleteFromLocalFileSystem(photo: Photo): PhotoResult = runCatching {
        val file = File(imagesDir, photo.fileName)
        if (!file.exists()) return PhotoResult.Success

        val result = file.delete()

        return if (result) PhotoResult.Success else PhotoResult.Error("Could not delete file.")
    }.getOrElse {
        PhotoResult.Error(it.message)
    }

    private fun ensureDirectoryExists() {
        if (!imagesDir.exists() && !imagesDir.mkdirs()) {
            throw FileSystemException(
                imagesDir,
                reason = "Failed to create directory"
            )
        }
    }

    private inline fun <T> Bitmap.use(block: (Bitmap) -> T): T {
        return try {
            block(this)
        } finally {
            recycle()
        }
    }
}