package net.damschen.swatchit.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.resultWrappers.CalculationResult
import net.damschen.swatchit.domain.resultWrappers.ValidationResult
import kotlin.math.ceil

class Swatch private constructor(
    val needleSize: KnittingNeedleSize,
    val pattern: Pattern? = null,
    val yarn: Yarn? = null,
    val notes: Notes? = null,
    val createdAt: EpochMillis,
    val id: SwatchId? = null,
    val name: Name? = null,
    private val _photo: Photo? = null,
    private val _gauge: Gauge? = null,
    private val _measurements: List<Measurement> = emptyList()
) {
    val gauge: Gauge? get() = _gauge
    val photo: Photo? get() = _photo
    val measurements get() = _measurements.toList()
    private val defaultGaugeSize = 10.0

    companion object {
        fun create(
            needleSize: KnittingNeedleSize,
            pattern: Pattern? = null,
            yarn: Yarn? = null,
            notes: Notes? = null,
            createdAt: EpochMillis,
            id: SwatchId? = null,
            name: Name? = null
        ): Swatch {
            return Swatch(
                needleSize = needleSize,
                pattern = pattern,
                yarn = yarn,
                notes = notes,
                createdAt = createdAt,
                id = id,
                name = name
            )
        }
    }

    private fun copy(
        needleSize: KnittingNeedleSize = this.needleSize,
        pattern: Pattern? = this.pattern,
        yarn: Yarn? = this.yarn,
        notes: Notes? = this.notes,
        createdAt: EpochMillis = this.createdAt,
        id: SwatchId? = this.id,
        name: Name? = this.name,
        photo: Photo? = this.photo,
        gauge: Gauge? = this.gauge,
        measurements: List<Measurement> = this.measurements
    ): Swatch {
        return Swatch(
            needleSize = needleSize,
            pattern = pattern,
            yarn = yarn,
            notes = notes,
            createdAt = createdAt,
            id = id,
            name = name,
            _photo = photo,
            _gauge = gauge,
            _measurements = measurements
        )
    }

    fun withUpdatedGauge(gauge: Gauge): Swatch = copy(gauge = gauge)

    fun withUpdatedPhoto(photo: Photo): Swatch = copy(photo = photo)

    fun withoutPhoto(): Swatch = copy(photo = null)

    fun withNewMeasurements(measurements: List<Measurement>): Swatch =
        copy(measurements = measurements)

    fun withMeasurement(measurement: Measurement): Swatch =
        copy(measurements = _measurements + measurement)

    fun withoutMeasurementAt(index: Int): Swatch {
        require(index in _measurements.indices) { "Invalid index" }
        return copy(measurements = _measurements.filterIndexed { i, _ -> i != index })
    }

    fun calculateGaugeFromMeasurements(): CalculationResult<Gauge> {
        if (measurements.isEmpty()) return CalculationResult.Error(R.string.no_measurements)

        val (stsMeasurements, rowMeasurements) = measurements.partition { it.measurementType == MeasurementType.Stitches }

        if (stsMeasurements.isEmpty()) return CalculationResult.Error(R.string.no_stitches)
        if (rowMeasurements.isEmpty()) return CalculationResult.Error(R.string.no_rows)

        val accumulatedStitches =
            stsMeasurements.sumOf { defaultGaugeSize / it.size.value * it.gaugeCount.value }
        val accumulatedRows =
            rowMeasurements.sumOf { defaultGaugeSize / it.size.value * it.gaugeCount.value }

        val stitches = ceil(accumulatedStitches / stsMeasurements.size).toInt()
        val rows = ceil(accumulatedRows / rowMeasurements.size).toInt()

        if (GaugeCount.validate(stitches) is ValidationResult.Error || GaugeCount.validate(rows) is ValidationResult.Error)
            return CalculationResult.Error(R.string.count_too_high)

        return CalculationResult.Success(
            Gauge(
                nrOfStitches = GaugeCount(stitches),
                nrOfRows = GaugeCount(rows),
                size = GaugeSize(defaultGaugeSize)
            )
        )
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is Swatch) return false

        return id == other.id
    }

    override fun hashCode(): Int {
        return id?.hashCode() ?: 0
    }
}