package net.damschen.swatchit.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.domain.resultWrappers.ValidationResult

data class GaugeSize(val value: Double) {
    val unit: LengthUnit = LengthUnit.CM

    init {
        require(validate(value) is ValidationResult.Success) { "GaugeSize invalid" }
    }

    companion object {
        fun validate(value: Double): ValidationResult {
            return when {
                value < MIN -> ValidationResult.Error(R.string.must_be_greater_than_zero)
                value > MAX -> ValidationResult.Error(R.string.max_99)
                else -> ValidationResult.Success
            }
        }

        fun validate(value: String): ValidationResult {
            val trimmed = value.trim()
            if (trimmed.isBlank()) return ValidationResult.Error(R.string.mandatory)
            if (trimmed.substringAfterLast(
                    '.',
                    ""
                ).length > MAX_DECIMAL_NUMBERS
            ) return ValidationResult.Error(R.string.max_1_decimal)
            val gaugeSize =
                trimmed.toDoubleOrNull() ?: return ValidationResult.Error(R.string.invalid_number)
            val validation = validate(gaugeSize)
            return validation
        }

        fun create(value: String): GaugeSize {
            val trimmed = value.trim()
            require(validate(trimmed) is ValidationResult.Success) { "Size invalid" }
            val numericValue = trimmed.toDouble()
            return GaugeSize(numericValue)
        }
    }

    override fun toString(): String = value.toPlainStringWithoutTrailingZeros()
}

private const val MIN = 0.1
private const val MAX = 99
private const val MAX_DECIMAL_NUMBERS = 1

enum class LengthUnit() {
    CM
}

