package net.damschen.swatchit.test.ui.viewmodels

import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.test.UnconfinedTestDispatcher
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.MeasurementType
import net.damschen.swatchit.shared.testhelpers.testdata.SwatchTestData
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.test.testHelpers.MainDispatcherRule
import net.damschen.swatchit.ui.enums.CountType
import net.damschen.swatchit.ui.enums.toCountType
import net.damschen.swatchit.ui.models.GaugeCalculationState
import net.damschen.swatchit.ui.models.GaugeState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.MeasurementListItem
import net.damschen.swatchit.ui.models.MeasurementsState
import net.damschen.swatchit.ui.viewmodels.MeasurementViewModel
import org.junit.Assert.assertEquals
import org.junit.Assert.assertFalse
import org.junit.Assert.assertNotNull
import org.junit.Assert.assertNull
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test

class MeasurementViewModelTests {
    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    private var repo = FakeRepo()
    private lateinit var sut: MeasurementViewModel

    @Before
    fun initSut() {
        repo = FakeRepo()
        sut = MeasurementViewModel(repo.defaultId, repo)
    }

    @Test
    fun onCountChanged_validCount_updatesState() {
        sut.onCountChanged("13")

        assertEquals("13", sut.measurementFormState.value.count.value)
        assertTrue(sut.measurementFormState.value.count.isValid())
    }

    @Test
    fun onCountChanged_invalidCount_updatesState() {
        sut.onCountChanged("abc")

        assertEquals("abc", sut.measurementFormState.value.count.value)
        assertFalse(sut.measurementFormState.value.count.isValid())
    }

    @Test
    fun onSizeChanged_validSize_updatesState() {
        sut.onSizeChanged("13")

        assertEquals("13", sut.measurementFormState.value.size.value)
        assertTrue(sut.measurementFormState.value.size.isValid())
    }

    @Test
    fun onSizeChanged_invalidSize_updatesState() {
        sut.onSizeChanged("-1")

        assertEquals("-1", sut.measurementFormState.value.size.value)
        assertFalse(sut.measurementFormState.value.size.isValid())
    }

    @Test
    fun onCountTypeChanged_validCountType_updatesState() {
        sut.onCountTypeChanged(CountType.Rows)

        assertEquals(CountType.Rows, sut.measurementFormState.value.type)
    }

    @Test
    fun loadSwatch_existingId_loadsList() {
        assertTrue(sut.loadState.value is LoadState.Success)
        assertEquals(
            MeasurementsState(
                listOf(
                    MeasurementListItem(
                        repo.defaultMeasurement.gaugeCount.value,
                        repo.defaultMeasurement.size.value,
                        repo.defaultMeasurement.measurementType.toCountType()
                    )
                )
            ), sut.measurementsState.value
        )
    }

    @Test
    fun loadSwatch_nonExistingId_setsLoadStateToNotFound() {
        repo.returnNull = true
        sut = MeasurementViewModel(13, repo)

        assertTrue(sut.loadState.value is LoadState.NotFound)
        assertEquals(MeasurementsState(emptyList()), sut.measurementsState.value)
    }

    @Test
    fun loadSwatch_repoReturnsError_setsLoadStateToError() {
        repo.returnError = true
        sut = MeasurementViewModel(repo.defaultId, repo)

        assertTrue(sut.loadState.value is LoadState.Error)
        assertEquals(MeasurementsState(emptyList()), sut.measurementsState.value)
    }

    @Test
    fun addMeasurement_swatchWithGauge_callsRepoWithUnchangedGauge() {
        val initialSwatch = SwatchTestData.from(repo.swatchToReturn())!!

        sut.onCountChanged("13")
        sut.onSizeChanged("20")
        sut.onCountTypeChanged(CountType.Stitches)
        sut.addMeasurement()

        val newMeasurements = ArrayList(initialSwatch.measurements)
        newMeasurements.add(
            Measurement(
                GaugeCount(13),
                GaugeSize(20.0),
                MeasurementType.Stitches
            )
        )
        val expected = initialSwatch.copy(measurements = newMeasurements)

        assertEquals(expected, SwatchTestData.from(repo.updatedSwatch))
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun addMeasurement_repoReturnsSuccess_emitSuccess() = runTest {
        sut.onCountChanged("13")
        sut.onSizeChanged("20")
        sut.onCountTypeChanged(CountType.Stitches)

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.savedSuccessfully.collect {
                values.add(it)
            }
        }

        sut.addMeasurement()

        assertTrue(values.last())
    }

    @Test
    fun init_uninitializedInput_errorIdsSetInState() = runTest {
        assertNotNull(sut.measurementFormState.value.count.errorMessageId)
        assertNotNull(sut.measurementFormState.value.size.errorMessageId)
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun addMeasurement_invalidInput_setsError() = runTest {
        sut.onCountChanged("-13")
        sut.onSizeChanged("20")

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.savedSuccessfully.collect {
                values.add(it)
            }
        }

        sut.addMeasurement()

        assertFalse(values.last())
    }

    @Test
    fun addMeasurement_repoReturnsSuccess_MeasurementAddedToStateList() {
        sut.onCountChanged("13")
        sut.onSizeChanged("20")
        sut.onCountTypeChanged(CountType.Stitches)
        sut.addMeasurement()

        assertEquals(
            MeasurementListItem(13, 20.0, CountType.Stitches),
            sut.measurementsState.value.items.last()
        )
    }

    @Test
    fun addMeasurement_twoConsecutiveMeasurements_MeasurementIsNotOverwritten() {
        sut.onCountChanged("13")
        sut.onSizeChanged("20")
        sut.onCountTypeChanged(CountType.Stitches)
        sut.addMeasurement()
        sut.onCountChanged("13")
        sut.onSizeChanged("20")
        sut.onCountTypeChanged(CountType.Stitches)
        sut.addMeasurement()

        val expectedMeasurements = listOf(
            repo.defaultMeasurement,
            Measurement(GaugeCount(13), GaugeSize(20.0), MeasurementType.Stitches),
            Measurement(GaugeCount(13), GaugeSize(20.0), MeasurementType.Stitches)
        )

        assertEquals(expectedMeasurements, repo.updatedSwatch!!.measurements)
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun addMeasurement_repoReturnsErrorDuringUpdate_setsErrorState() = runTest {
        sut.onCountChanged("13")
        sut.onSizeChanged("20")
        sut.onCountTypeChanged(CountType.Stitches)
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.savedSuccessfully.collect {
                values.add(it)
            }
        }

        repo.returnErrorDuringUpdate = true

        sut.addMeasurement()

        assertFalse(values.last())
    }

    @Test
    fun deleteMeasurementAt_existingMeasurement_callsRepo() {
        val initialSwatch = SwatchTestData.from(repo.swatchToReturn())!!

        sut.deleteMeasurementAt(0)

        val expected = initialSwatch.copy(measurements = initialSwatch.measurements.dropLast(1))

        assertEquals(expected, SwatchTestData.from(repo.updatedSwatch))
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun deleteMeasurementAt_repoReturnsSuccess_emitSuccess() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.deletedSuccessfully.collect {
                values.add(it)
            }
        }

        sut.deleteMeasurementAt(0)

        assertTrue(values.last())
    }

    @Test
    fun deleteMeasurementAt_repoReturnsSuccess_MeasurementDeletedFromStateList() {
        sut.deleteMeasurementAt(0)

        assertEquals(0, sut.measurementsState.value.items.count())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun deleteMeasurementAt_repoReturnsErrorDuringUpdate_setsErrorState() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.deletedSuccessfully.collect {
                values.add(it)
            }
        }

        repo.returnErrorDuringUpdate = true
        sut.deleteMeasurementAt(0)

        assertFalse(values.last())
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun deleteMeasurementAt_nonExistingMeasurement_setsErrorState() = runTest {
        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.deletedSuccessfully.collect {
                values.add(it)
            }
        }

        sut.deleteMeasurementAt(3)

        assertFalse(values.last())
    }

    @Test
    fun deleteMeasurementAt_twoEqualItems_oneMeasurementDeletedFromStateList() {
        val swatch = repo.swatchToReturn()!!.withMeasurement(repo.defaultMeasurement)
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)


        sut.deleteMeasurementAt(0)

        assertEquals(1, sut.measurementsState.value.items.count())
    }

    @Test
    fun calculateGauge_validMeasurements_updatesGaugeState() {
        val swatch = repo.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)

        sut.calculateGauge()
        assertEquals(
            GaugeCalculationState(
                GaugeState(34, 34, 10.0),
                null
            ), sut.gaugeCalculationState.value
        )
    }

    @Test
    fun calculateGauge_noMeasurements_setsErrorIdInGaugeState() {
        val swatch = repo.swatchToReturn()!!
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)

        sut.calculateGauge()
        assertNotNull(sut.gaugeCalculationState.value.errorMessageId)
        assertNull(sut.gaugeCalculationState.value.gaugeState)
    }

    @Test
    fun saveGaugeCalculationState_gaugeHasBeenCalculated_callsRepo() {
        val swatch = repo.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)

        sut.calculateGauge()
        sut.saveGaugeCalculationState()

        assertEquals(
            Gauge(GaugeCount(34), GaugeCount(34), GaugeSize(10.0)),
            repo.updatedSwatch!!.gauge
        )
    }

    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeCalculationState_gaugeHasBeenCalculated_emitsSuccess() = runTest {
        val swatch = repo.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)

        sut.calculateGauge()

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveGaugeCalculationState()

        assertTrue(values.last())
    }


    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeCalculationState_gaugeHasNotBeenCalculated_emitsError() = runTest {
        val swatch = repo.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveGaugeCalculationState()

        assertFalse(values.last())
    }


    @OptIn(ExperimentalCoroutinesApi::class)
    @Test
    fun saveGaugeCalculationState_repoReturnsError_emitsError() = runTest {
        val swatch = repo.swatchToReturn()!!.withNewMeasurements(
            listOf(
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Rows),
                Measurement(GaugeCount(34), GaugeSize(10.0), MeasurementType.Stitches)
            )
        )
        repo.swatchToReturn = { swatch }
        sut = MeasurementViewModel(repo.defaultId, repo)

        sut.calculateGauge()
        repo.returnErrorDuringUpdate = true

        val values = mutableListOf<Boolean>()
        backgroundScope.launch(UnconfinedTestDispatcher(testScheduler)) {
            sut.gaugeSuccessfullySaved.collect {
                values.add(it)
            }
        }

        sut.saveGaugeCalculationState()

        assertFalse(values.last())
    }
}