package net.damschen.swatchit.test.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.MeasurementType
import net.damschen.swatchit.domain.aggregates.swatch.Name
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.resultWrappers.CalculationResult
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.shared.testhelpers.testdata.SwatchTestData
import org.junit.Assert.assertEquals
import org.junit.Assert.assertFalse
import org.junit.Assert.assertThrows
import org.junit.Assert.assertTrue
import org.junit.Test
import java.time.LocalDate
import java.time.ZoneId
import java.util.UUID

class SwatchTests {

    @Test
    fun withUpdatedGauge_swatchWithGaugeAndMeasurements_copiesMeasurementsAndUpdatesGauge() {
        val oldSwatch =
            createSut().withMeasurement(defaultMeasurement).withUpdatedPhoto(defaultPhoto)
        val updatedGauge = Gauge(GaugeCount(2), GaugeCount(5), GaugeSize(13.0))
        val newSwatch = oldSwatch.withUpdatedGauge(updatedGauge)

        assertEquals(
            SwatchTestData.from(oldSwatch)?.copy(gauge = updatedGauge),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withUpdatedGauge_swatchWithoutGaugeWithMeasurements_copiesMeasurementsAndUpdatesGauge() {
        val oldSwatch =
            createSut(null).withMeasurement(defaultMeasurement).withUpdatedPhoto(defaultPhoto)
        val updatedGauge = Gauge(GaugeCount(2), GaugeCount(5), GaugeSize(13.0))
        val newSwatch = oldSwatch.withUpdatedGauge(updatedGauge)

        assertEquals(
            SwatchTestData.from(oldSwatch)?.copy(gauge = updatedGauge),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withUpdatedPhoto_swatchWithPhoto_replacesPhoto() {
        val oldSwatch =
            createSut().withMeasurement(defaultMeasurement).withUpdatedPhoto(defaultPhoto)
        val uuid = UUID.fromString("1c46b56a-22a2-420f-b481-45e641a39088")
        val updatedPhoto = Photo(uuid)
        val newSwatch = oldSwatch.withUpdatedPhoto(updatedPhoto)

        assertEquals(
            SwatchTestData.from(oldSwatch)?.copy(photo = updatedPhoto),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withUpdatedPhoto_swatchWithoutPhoto_addsPhoto() {
        val oldSwatch = createSut(null).withMeasurement(defaultMeasurement)
        val uuid = UUID.fromString("1c46b56a-22a2-420f-b481-45e641a39088")
        val updatedPhoto = Photo(uuid)
        val newSwatch = oldSwatch.withUpdatedPhoto(updatedPhoto)

        assertEquals(
            SwatchTestData.from(oldSwatch)?.copy(photo = updatedPhoto),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withoutPhoto_swatchWithPhoto_removesPhoto() {
        val uuid = UUID.fromString("1c46b56a-22a2-420f-b481-45e641a39088")
        val updatedPhoto = Photo(uuid)
        val oldSwatch =
            createSut(null).withMeasurement(defaultMeasurement).withUpdatedPhoto(updatedPhoto)
        val newSwatch = oldSwatch.withoutPhoto()

        assertEquals(
            SwatchTestData.from(oldSwatch)?.copy(photo = null),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withoutPhoto_swatchWithoutPhoto_returnsUnchangedSwatch() {
        val oldSwatch = createSut(null).withMeasurement(defaultMeasurement)
        val newSwatch = oldSwatch.withoutPhoto()

        assertEquals(
            SwatchTestData.from(oldSwatch),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withNewMeasurements_swatchWithoutExistingMeasurements_replacesMeasurements() {
        val oldSwatch = createSut().withUpdatedPhoto(defaultPhoto)
        val newMeasurements =
            listOf(defaultMeasurement, defaultMeasurement.copy(gaugeCount = GaugeCount(34)))
        val newSwatch = oldSwatch.withNewMeasurements(newMeasurements)

        assertEquals(
            SwatchTestData.from(oldSwatch)!!.copy(measurements = newMeasurements),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withNewMeasurements_swatchWithExistingMeasurements_replacesMeasurements() {
        var sut =
            createSut().withNewMeasurements(listOf(defaultMeasurement.copy(gaugeCount = GaugeCount(2))))

        val newMeasurements =
            listOf(defaultMeasurement, defaultMeasurement.copy(gaugeCount = GaugeCount(34)))

        sut = sut.withNewMeasurements(newMeasurements)

        assertEquals(newMeasurements, sut.measurements)
    }

    @Test
    fun withMeasurement_swatchWithoutExistingMeasurements_addsMeasurement() {
        val oldSwatch = createSut()
        val newSwatch = createSut().withMeasurement(defaultMeasurement)

        assertEquals(
            SwatchTestData.from(oldSwatch)!!.copy(measurements = listOf(defaultMeasurement)),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun withMeasurement_swatchWithExistingMeasurements_addsMeasurement() {
        var sut = createSut().withMeasurement(defaultMeasurement)

        sut = sut.withMeasurement(defaultMeasurement)

        assertEquals(listOf(defaultMeasurement, defaultMeasurement), sut.measurements)
    }

    @Test
    fun withoutMeasurementAt_measurementDoesNotExist_throws() {
        val sut = createSut()
        assertThrows(IllegalArgumentException::class.java) {
            sut.withoutMeasurementAt(0)
        }
    }

    @Test
    fun withoutMeasurementAt_measurementExists_removesMeasurement() {
        val oldSwatch = createSut().withMeasurement(defaultMeasurement)

        val newSwatch = oldSwatch.withoutMeasurementAt(0)

        assertEquals(
            SwatchTestData.from(oldSwatch)!!.copy(measurements = emptyList()),
            SwatchTestData.from(newSwatch)
        )
    }

    @Test
    fun calculateGaugeFromMeasurements_noMeasurementsInList_ReturnsErrorId() {
        val sut = createSut()

        val result = sut.calculateGaugeFromMeasurements()

        assertTrue(result is CalculationResult.Error)
        assertEquals(R.string.no_measurements, (result as CalculationResult.Error).errorMessageId)
    }

    @Test
    fun calculateGaugeFromMeasurements_onlyRowMeasurementsInList_ReturnsErrorId() {
        val sut = createSut().withNewMeasurements(
            listOf(
                Measurement(GaugeCount(15), GaugeSize(26.0), MeasurementType.Rows)
            )
        )

        val result = sut.calculateGaugeFromMeasurements()

        assertTrue(result is CalculationResult.Error)
        assertEquals(R.string.no_stitches, (result as CalculationResult.Error).errorMessageId)
    }

    @Test
    fun calculateGaugeFromMeasurements_onlyStitchesMeasurementsInList_ReturnsErrorId() {
        val sut = createSut().withNewMeasurements(
            listOf(
                Measurement(GaugeCount(15), GaugeSize(26.0), MeasurementType.Stitches)
            )
        )

        val result = sut.calculateGaugeFromMeasurements()

        assertTrue(result is CalculationResult.Error)
        assertEquals(R.string.no_rows, (result as CalculationResult.Error).errorMessageId)
    }

    @Test
    fun calculateGaugeFromMeasurements_stitchesAndRowsInMeasurementsList_ReturnsResult() {
        val sut = createSut().withNewMeasurements(
            listOf(
                Measurement(GaugeCount(15), GaugeSize(26.0), MeasurementType.Stitches),
                Measurement(GaugeCount(9), GaugeSize(12.5), MeasurementType.Stitches),
                Measurement(GaugeCount(15), GaugeSize(26.0), MeasurementType.Rows),
                Measurement(GaugeCount(27), GaugeSize(34.5), MeasurementType.Rows),
                Measurement(GaugeCount(22), GaugeSize(6.0), MeasurementType.Rows)
            )
        )

        val gauge = sut.calculateGaugeFromMeasurements()

        assertTrue(gauge is CalculationResult.Success)
        assertEquals(
            Gauge(GaugeCount(7), GaugeCount(17), GaugeSize(10.0)),
            (gauge as CalculationResult.Success).data
        )
    }

    @Test
    fun calculateGaugeFromMeasurements_stitchesCalculationLeadsToTooHighValue_ReturnsErrorId() {
        val sut = createSut().withNewMeasurements(
            listOf(
                Measurement(GaugeCount(99), GaugeSize(1.0), MeasurementType.Stitches),
                Measurement(GaugeCount(2), GaugeSize(1.0), MeasurementType.Rows)
            )
        )

        val result = sut.calculateGaugeFromMeasurements()

        assertTrue(result is CalculationResult.Error)
        assertEquals(R.string.count_too_high, (result as CalculationResult.Error).errorMessageId)
    }

    @Test
    fun calculateGaugeFromMeasurements_rowsCalculationLeadsToTooHighValue_ReturnsErrorId() {
        val sut = createSut().withNewMeasurements(
            listOf(
                Measurement(GaugeCount(2), GaugeSize(1.0), MeasurementType.Stitches),
                Measurement(GaugeCount(99), GaugeSize(1.0), MeasurementType.Rows)
            )
        )

        val result = sut.calculateGaugeFromMeasurements()

        assertTrue(result is CalculationResult.Error)
        assertEquals(R.string.count_too_high, (result as CalculationResult.Error).errorMessageId)
    }

    @Test
    fun equals_swatchesWithEqualIds_returnsTrue() {
        val first = createSut(null)
        val second = createSut(gauge = defaultGauge.copy(nrOfRows = GaugeCount(21)))

        assertTrue(first == second)
    }

    @Test
    fun equals_swatchesWithDifferentIds_returnsFalse() {
        val first = createSut(null)
        val second = createSut(id = SwatchId(34))

        assertFalse(first == second)
    }

    @Test
    fun create_swatchWithAllFieldsSet_createsInstance() {
        val swatch = Swatch.create(
            needleSize = defaultNeedleSize,
            pattern = defaultPattern,
            yarn = defaultYarn,
            notes = defaultNotes,
            createdAt = defaultCreatedAt,
            id = defaultId,
            name = defaultName
        )

        assertEquals(defaultId, swatch.id)
        assertEquals(defaultPattern, swatch.pattern)
        assertEquals(defaultName, swatch.name)
        assertEquals(defaultNotes, swatch.notes)
        assertEquals(defaultCreatedAt, swatch.createdAt)
        assertEquals(defaultYarn, swatch.yarn)
        assertEquals(defaultNeedleSize, swatch.needleSize)
    }
}

private val defaultMeasurement =
    Measurement(
        gaugeCount = GaugeCount(23),
        size = GaugeSize(12.5),
        measurementType = MeasurementType.Rows
    )

private val defaultNeedleSize = KnittingNeedleSize.SIZE_2_5
private val defaultPattern = Pattern.create("Stockinette")
private val defaultGauge =
    Gauge(GaugeCount(value = 20), GaugeCount(30), GaugeSize(10.0))
private val defaultYarn = Yarn.create(
    Name.create("Yarn Name"), Name.create("Yarn Manufacturer")
)
private val defaultNotes = Notes.create("Test notes!")
private val defaultCreatedAt = EpochMillis(
    LocalDate.of(2025, 2, 17).atStartOfDay(
        ZoneId.of("UTC")
    ).toInstant().toEpochMilli()
)
private val defaultName = Name.create("Test Name")

private val defaultId = SwatchId(1)

private val defaultPhoto = Photo.create(FakeUUIDProvider())

private fun createSut(gauge: Gauge? = defaultGauge, id: SwatchId = defaultId): Swatch {
    val swatch = Swatch.create(
        needleSize = defaultNeedleSize,
        pattern = defaultPattern,
        yarn = defaultYarn,
        notes = defaultNotes,
        createdAt = defaultCreatedAt,
        id = id,
        name = defaultName,
    )

    return gauge?.let { swatch.withUpdatedGauge(gauge) } ?: swatch
}