package net.damschen.swatchit.test.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.aggregates.swatch.Size
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import org.junit.Assert.assertEquals
import org.junit.Assert.assertFalse
import org.junit.Assert.assertNull
import org.junit.Assert.assertThrows
import org.junit.Assert.assertTrue
import org.junit.Test

class SizeTests {
    @Test
    fun init_validInput_createsSize() {
        val value = 13.2
        val size = Size(value)
        assertEquals(value, size.value, 1e-2)
    }

    @Test
    fun init_negativeNumber_throws() {
        assertThrows(IllegalArgumentException::class.java) {
            Size(-1.3)
        }
    }

    @Test
    fun validate_validInput_returnsIsValid() {
        val validationResult = Size.validate("13.2")
        assertTrue(validationResult.isValid)
    }

    @Test
    fun validate_validInputWithTrailingWhitespace_returnsIsValid() {
        val validationResult = Size.validate("  13.2   ")
        assertTrue(validationResult.isValid)
    }

    @Test
    fun validate_negativeNumber_returnsErrorMessageId() {
        val validationResult = Size.validate("-1")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.must_be_greater_than_zero, validationResult.errorMessageId)
    }

    @Test
    fun validate_zero_returnsErrorMessageId() {
        val validationResult = Size.validate("0")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.must_be_greater_than_zero, validationResult.errorMessageId)
    }

    @Test
    fun validate_valueNaN_returnErrorMessageId() {
        val validationResult = Size.validate("abc")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.invalid_number, validationResult.errorMessageId)
    }

    @Test
    fun validate_valueEmpty_returnIsValid() {
        val validationResult = Size.validate("")
        assertTrue(validationResult.isValid)
    }

    @Test
    fun validate_twoDigitsAfterDecimal_returnErrorMessageId() {
        val validationResult = Size.validate("13.24")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.max_1_decimal, validationResult.errorMessageId)
    }

    @Test
    fun validate_greaterThanMaxValue_returnErrorMessageId() {
        val validationResult =
            Size.validate((Double.MAX_VALUE + 0.1).toPlainStringWithoutTrailingZeros())
        assertFalse(validationResult.isValid)
        assertEquals(R.string.number_too_big, validationResult.errorMessageId)
    }

    @Test
    fun create_validInput_returnsInstance() {
        val size = Size.create("13.2")
        assertEquals(13.2, size!!.value, 1e-2)
    }

    @Test
    fun create_valueNaN_throws() {
        assertThrows(IllegalArgumentException::class.java) { Size.create("abc") }
    }

    @Test
    fun create_valueEmpty_returnsNull() {
        assertNull(Size.create(""))
    }

    @Test
    fun create_inputWithTrailingWhitespace_returnsInstance() {
        val size = Size.create("  13  ")
        assertEquals(13.0, size!!.value, 1e-2)
    }

    @Test
    fun toString_validValue_printsValueWithoutTrailingZeros() {
        val size = Size(50.0)
        assertEquals("50", size.toString())
    }
}