package net.damschen.swatchit.test.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import org.junit.Assert.assertEquals
import org.junit.Assert.assertFalse
import org.junit.Assert.assertThrows
import org.junit.Assert.assertTrue
import org.junit.Test

class GaugeCountTests {

    @Test
    fun init_validInput_createsCount() {
        val gaugeCount = GaugeCount(13)
        assertEquals(13, gaugeCount.value)
    }

    @Test
    fun init_negativeNumber_throws() {
        assertThrows(IllegalArgumentException::class.java) {
            GaugeCount(-1)
        }
    }

    @Test
    fun init_numberGreaterThan99_throws() {
        assertThrows(IllegalArgumentException::class.java) {
            GaugeCount(100)
        }
    }

    @Test
    fun validate_validInput_returnsIsValid() {
        val validationResult = GaugeCount.validate("13")
        assertTrue(validationResult.isValid)
    }

    @Test
    fun validate_validInputWithTrailingWhitespace_returnsIsValid() {
        val validationResult = GaugeCount.validate(" 13   ")
        assertTrue(validationResult.isValid)
    }

    @Test
    fun validate_negativeNumber_returnsErrorMessageId() {
        val validationResult = GaugeCount.validate("-1")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.must_be_greater_than_zero, validationResult.errorMessageId)
    }

    @Test
    fun validate_zero_returnsErrorMessageId() {
        val validationResult = GaugeCount.validate("0")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.must_be_greater_than_zero, validationResult.errorMessageId)
    }

    @Test
    fun validate_numberGreaterThan99_returnErrorMessageId() {
        val validationResult = GaugeCount.validate("100")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.max_99, validationResult.errorMessageId)
    }

    @Test
    fun validate_valueNaN_returnErrorMessageId() {
        val validationResult = GaugeCount.validate("abc")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.invalid_number, validationResult.errorMessageId)
    }

    @Test
    fun validate_valueEmpty_returnErrorMessageId() {
        val validationResult = GaugeCount.validate(" ")
        assertFalse(validationResult.isValid)
        assertEquals(R.string.mandatory, validationResult.errorMessageId)
    }

    @Test
    fun toString_validValue_printsValue() {
        val count = GaugeCount(12)
        assertEquals(12.toString(), count.toString())
    }

    @Test
    fun create_validInput_returnsInstance() {
        val count = GaugeCount.create("13")
        assertEquals(13, count.value)
    }

    @Test
    fun create_inputWithTrailingWhitespace_returnsInstance() {
        val count = GaugeCount.create(" 13  ")
        assertEquals(13, count.value)
    }

    @Test
    fun create_valueNaN_throws() {
        assertThrows(IllegalArgumentException::class.java) { GaugeCount.create("abc") }
    }
}