package net.damschen.swatchit.integrationTest.ui.screens

import androidx.compose.ui.test.assertTextContains
import androidx.compose.ui.test.isNotDisplayed
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performTextInput
import androidx.compose.ui.test.performTextReplacement
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import dagger.hilt.android.testing.HiltTestApplication
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.infrastructure.database.AppDatabase
import net.damschen.swatchit.infrastructure.database.SwatchDao
import net.damschen.swatchit.infrastructure.database.SwatchEntity
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.test.testHelpers.FakePhotoStorageService
import net.damschen.swatchit.ui.screens.SwatchDetailsScreen
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.After
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.annotation.Config
import org.robolectric.shadows.ShadowLog
import java.time.LocalDate
import java.time.ZoneId
import javax.inject.Inject
import javax.inject.Named

@RunWith(RobolectricTestRunner::class)
@HiltAndroidTest
@Config(application = HiltTestApplication::class)
class SwatchDetailsScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    @get:Rule
    var hiltRule = HiltAndroidRule(this)

    private lateinit var viewModel: EditSwatchViewModel

    @Inject
    @Named("test_db")
    lateinit var database: AppDatabase

    @Inject
    internal lateinit var swatchDao: SwatchDao

    @Inject
    internal lateinit var repository: SwatchRepository

    @Inject
    internal lateinit var photoStorageService: PhotoStorageService

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initDb() {
        hiltRule.inject()

        database.clearAllTables()
        viewModel = EditSwatchViewModel(
            1, repository, FakeDateTimeProvider(), photoStorageService,
            FakeUUIDProvider()
        )
    }

    @After
    fun closeDb() {
        database.close()
    }

    @Test
    fun editGaugeDialog_gaugeAdded_closesDialogAndUpdatesDetailsView() = runTest {
        swatchDao.insert(swatch = minimalSwatchEntity).toInt()
        viewModel = EditSwatchViewModel(
            1,
            repository,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag(nrOfStitchesInputTag).performTextInput("13")
        composeTestRule.onNodeWithTag(nrOfRowsInputTag).performTextInput("21")
        composeTestRule.onNodeWithTag(gaugeLengthInputTag).performTextInput("10")

        composeTestRule.onNodeWithTag(saveGaugeButtonTag).performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag(editGaugeDialogTag).isNotDisplayed()
        }

        composeTestRule.onNodeWithTag(nrOfStitchesValueTag)
            .assertTextContains("13", substring = true)
        composeTestRule.onNodeWithTag(nrOfRowsValueTag).assertTextContains("21", substring = true)
        composeTestRule.onNodeWithTag(gaugeLengthValueTag)
            .assertTextContains("10", substring = true)
    }

    @Test
    fun editGaugeDialog_existingGaugeUpdated_closesDialogAndUpdatesDetailsView() = runTest {
        swatchDao.insert(swatch = swatchEntityWithGauge).toInt()
        viewModel = EditSwatchViewModel(
            1,
            repository,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag(nrOfStitchesInputTag).performTextReplacement("13")

        composeTestRule.onNodeWithTag(saveGaugeButtonTag).performClick()

        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag(editGaugeDialogTag).isNotDisplayed()
        }

        composeTestRule.onNodeWithTag(nrOfStitchesValueTag)
            .assertTextContains("13", substring = true)
        composeTestRule.onNodeWithTag(nrOfRowsValueTag)
            .assertTextContains(swatchEntityWithGauge.nrOfRows.toString(), substring = true)
        composeTestRule.onNodeWithTag(gaugeLengthValueTag)
            .assertTextContains(
                swatchEntityWithGauge.gaugeLength!!.toPlainStringWithoutTrailingZeros(),
                substring = true
            )
    }
}


private val defaultCreatedAt = EpochMillis(
    LocalDate.of(2025, 2, 17).atStartOfDay(
        ZoneId.of("UTC")
    ).toInstant().toEpochMilli()
)

private val minimalSwatchEntity = SwatchEntity(
    needleSize = net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_5,
    yarnName = "",
    yarnManufacturer = "",
    nrOfStitches = null,
    nrOfRows = null,
    gaugeLength = null,
    createdAt = defaultCreatedAt.value,
    name = "",
    pattern = "",
    notes = "",
    photoUUID = null
)
private val swatchEntityWithGauge = SwatchEntity(
    needleSize = net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_5,
    yarnName = "",
    yarnManufacturer = "",
    nrOfStitches = 55,
    nrOfRows = 23,
    gaugeLength = 10.0,
    createdAt = defaultCreatedAt.value,
    name = "",
    pattern = "",
    notes = "",
    photoUUID = null
)

private const val editGaugeButtonTag = "EditGaugeButton"
private const val nrOfStitchesInputTag = "NrOfStitchesInput"
private const val nrOfRowsInputTag = "NrOfRowsInput"
private const val gaugeLengthInputTag = "GaugeLengthInput"
private const val saveGaugeButtonTag = "SaveGaugeButton"
private const val editGaugeDialogTag = "EditGaugeDialog"
private const val nrOfStitchesValueTag = "NrOfStitches"
private const val nrOfRowsValueTag = "NrOfRows"
private const val gaugeLengthValueTag = "GaugeLength"
