package net.damschen.swatchit.integrationTest.infrastrcuture.services

import android.content.Context
import android.graphics.Bitmap
import android.net.Uri
import dagger.hilt.android.qualifiers.ApplicationContext
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import dagger.hilt.android.testing.HiltTestApplication
import junit.framework.TestCase.assertFalse
import junit.framework.TestCase.assertTrue
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.infrastructure.resultWrappers.PhotoResult
import net.damschen.swatchit.infrastructure.services.AndroidPhotoStorageService
import net.damschen.swatchit.infrastructure.services.BitmapDecoder
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.shared.testhelpers.FakeBitmapDecoder
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.annotation.Config
import java.io.File
import java.io.FileOutputStream
import javax.inject.Inject

@RunWith(RobolectricTestRunner::class)
@HiltAndroidTest
@Config(application = HiltTestApplication::class, sdk = [33])
class PhotoStorageServiceIntegrationTest {

    @get:Rule
    var hiltRule = HiltAndroidRule(this)

    lateinit var photoStorageService: PhotoStorageService
    lateinit var imagesDir: File

    @Inject
    @ApplicationContext
    lateinit var context: Context

    @Inject
    lateinit var bitmapDecoder: BitmapDecoder

    @Before
    fun setup() {
        hiltRule.inject()
        photoStorageService = AndroidPhotoStorageService(context, bitmapDecoder)
        imagesDir = File(context.filesDir, "images")
    }

    @Test
    fun copyToLocalFileSystem_createsImagesDirectory() = runTest {
        val testImageUri = createTestImageUri(context)
        val photo = Photo(FakeUUIDProvider.defaultUUID)

        val result = photoStorageService.copyToLocalFileSystem(testImageUri, photo)

        assertTrue(imagesDir.exists())
        assertTrue(imagesDir.isDirectory)
        assertTrue(result is PhotoResult.Success)
    }

    @Test
    fun copyToLocalFileSystem_savesImageWithCorrectFilename() = runTest {
        val testImageUri = createTestImageUri(context)
        val photo = Photo(FakeUUIDProvider.defaultUUID)

        val result = photoStorageService.copyToLocalFileSystem(testImageUri, photo)

        val savedFile = File(context.filesDir, "images/${FakeUUIDProvider.defaultUUID}.jpg")
        assertTrue(savedFile.exists())
        assertTrue(result is PhotoResult.Success)
    }

    @Test
    fun copyToLocalFileSystem_bitmapDecoderThrows_returnsErrorResult() = runTest {
        val testImageUri = createTestImageUri(context)
        val photo = Photo(FakeUUIDProvider.defaultUUID)
        (bitmapDecoder as FakeBitmapDecoder).shouldThrow = true

        val result = photoStorageService.copyToLocalFileSystem(testImageUri, photo)

        assertTrue(result is PhotoResult.Error)
    }

    @Test
    fun deleteFromLocalFileSystem_imageExists_removesImage() {
        createTestImageFile(context)
        val photo = Photo(FakeUUIDProvider.defaultUUID)

        val result = photoStorageService.deleteFromLocalFileSystem(photo)

        val savedFile = File(context.filesDir, "images/${FakeUUIDProvider.defaultUUID}.jpg")
        assertFalse(savedFile.exists())
        assertTrue(result is PhotoResult.Success)
    }

    @Test
    fun deleteFromLocalFileSystem_imageDoesNotExist_returnsSuccess() {
        val photo = Photo(FakeUUIDProvider.defaultUUID)

        val result = photoStorageService.deleteFromLocalFileSystem(photo)

        val savedFile = File(context.filesDir, "images/${FakeUUIDProvider.defaultUUID}.jpg")
        assertFalse(savedFile.exists())
        assertTrue(result is PhotoResult.Success)
    }
}

private fun createTestImageUri(context: Context): Uri {
    val bitmap = Bitmap.createBitmap(50, 50, Bitmap.Config.ARGB_8888)
    bitmap.eraseColor(android.graphics.Color.BLUE)

    val tempFile = File(context.cacheDir, "temp_${System.currentTimeMillis()}.png")
    FileOutputStream(tempFile).use { out ->
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, out)
    }
    bitmap.recycle()

    return Uri.fromFile(tempFile)
}

private fun createTestImageFile(context: Context) {
    val bitmap = Bitmap.createBitmap(50, 50, Bitmap.Config.ARGB_8888)
    bitmap.eraseColor(android.graphics.Color.BLUE)
    val dir = File(context.filesDir, "images").apply { mkdirs() }

    val file = File(dir, "${FakeUUIDProvider.defaultUUID}.jpg")
    FileOutputStream(file).use { out ->
        bitmap.compress(Bitmap.CompressFormat.PNG, 100, out)
    }
    bitmap.recycle()
}