package net.damschen.swatchit.ui.viewmodels

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.domain.resultWrappers.CalculationResult
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult
import net.damschen.swatchit.ui.enums.CountType
import net.damschen.swatchit.ui.models.GaugeCalculationState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.MeasurementFormState
import net.damschen.swatchit.ui.models.MeasurementsState
import net.damschen.swatchit.ui.models.SwatchState
import net.damschen.swatchit.ui.models.ValidatedInput

@HiltViewModel(assistedFactory = MeasurementViewModel.Factory::class)
class MeasurementViewModel @AssistedInject constructor(
    @Assisted private val swatchId: Int, private val swatchRepository: SwatchRepository
) : ViewModel() {
    private val _loadState = MutableStateFlow<LoadState>(LoadState.Initial)
    val loadState: StateFlow<LoadState> = _loadState.asStateFlow()

    private val _gaugeCalculationState = MutableStateFlow(
        GaugeCalculationState(null, null)
    )
    val gaugeCalculationState: StateFlow<GaugeCalculationState> =
        _gaugeCalculationState.asStateFlow()

    private val _savedSuccessfully = MutableSharedFlow<Boolean>()
    val savedSuccessfully: SharedFlow<Boolean> = _savedSuccessfully.asSharedFlow()

    private val _deletedSuccessfully = MutableSharedFlow<Boolean>()
    val deletedSuccessfully: SharedFlow<Boolean> = _deletedSuccessfully.asSharedFlow()

    private val _gaugeSuccessfullySaved = MutableSharedFlow<Boolean>()
    val gaugeSuccessfullySaved: SharedFlow<Boolean> = _gaugeSuccessfullySaved.asSharedFlow()

    private val defaultCountType = CountType.Stitches
    private val _measurementFormState = MutableStateFlow(
        MeasurementFormState(
            ValidatedInput(), ValidatedInput(), defaultCountType
        )
    )
    val measurementFormState: StateFlow<MeasurementFormState> = _measurementFormState

    private var _measurementsState = MutableStateFlow(MeasurementsState(emptyList()))
    val measurementsState: StateFlow<MeasurementsState> = _measurementsState

    private var _swatchState = MutableStateFlow<SwatchState?>(null)

    init {
        initFormState()
        loadSwatch()
    }

    @AssistedFactory
    interface Factory {
        fun create(swatchId: Int): MeasurementViewModel
    }

    fun loadSwatch() {
        viewModelScope.launch {
            _loadState.value = LoadState.Loading
            when (val result = swatchRepository.get(swatchId)) {
                is DatabaseResult.Success -> {
                    result.data?.let { swatch ->
                        _swatchState.value = SwatchState.from(swatch)
                        _measurementsState.value =
                            MeasurementsState.fromMeasurements(swatch.measurements)
                        _loadState.value = LoadState.Success
                    } ?: run {
                        _loadState.value = LoadState.NotFound
                    }
                }

                is DatabaseResult.Error -> {
                    _loadState.value = LoadState.Error(result.exception.message)
                }
            }
        }
    }

    fun addMeasurement() {
        viewModelScope.launch {
            val swatchState = _swatchState.value
            if (!_measurementFormState.value.isValid() || swatchState == null)
                return@launch _savedSuccessfully.emit(false)

            val swatch =
                swatchState.toSwatch()
                    .withMeasurement(_measurementFormState.value.toMeasurement())

            val updateResult = swatchRepository.update(swatch)
            val isSuccess = updateResult is DatabaseResult.Success

            if (isSuccess) {
                _swatchState.update { SwatchState.from(swatch) }
                _measurementsState.value =
                    MeasurementsState.fromMeasurements(swatch.measurements)
            }
            _savedSuccessfully.emit(isSuccess)
        }
    }

    fun deleteMeasurementAt(index: Int) {
        viewModelScope.launch {
            val currentSwatch = _swatchState.value ?: return@launch _savedSuccessfully.emit(false)

            var swatch: Swatch

            try {
                swatch = currentSwatch.toSwatch().withoutMeasurementAt(index)
            } catch (_: IllegalArgumentException) {
                return@launch _deletedSuccessfully.emit(false)
            }
            val updateResult = swatchRepository.update(swatch)
            val isSuccess = updateResult is DatabaseResult.Success
            if (isSuccess) {
                _swatchState.update { SwatchState.from(swatch) }
                _measurementsState.value =
                    MeasurementsState.fromMeasurements(swatch.measurements)
            }

            _deletedSuccessfully.emit(isSuccess)
        }
    }

    fun initFormState() {
        onCountChanged("")
        onSizeChanged("")
        onCountTypeChanged(defaultCountType)
    }

    fun onCountChanged(value: String) {
        val validationResult = GaugeCount.validate(value)
        _measurementFormState.update { currentState ->
            currentState.copy(
                count = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onSizeChanged(value: String) {
        val validationResult = GaugeSize.validate(value)
        _measurementFormState.update { currentState ->
            currentState.copy(
                size = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onCountTypeChanged(value: CountType) {
        _measurementFormState.update { currentState ->
            currentState.copy(type = value)
        }
    }

    fun calculateGauge() {
        val swatchState = _swatchState.value
        if (swatchState == null) {
            _gaugeCalculationState.value = GaugeCalculationState(null, null)
            return
        }

        val result = swatchState.toSwatch().calculateGaugeFromMeasurements()

        if (result is CalculationResult.Error)
            _gaugeCalculationState.value = GaugeCalculationState(null, result.errorMessageId)
        else if (result is CalculationResult.Success)
            _gaugeCalculationState.value = GaugeCalculationState.fromGauge(result.data)
    }

    fun saveGaugeCalculationState() {
        fun gaugeStateToSwatch(): Swatch? {
            return _swatchState.value?.let { oldSwatchState ->
                val gauge = _gaugeCalculationState.value.toGauge() ?: return null
                oldSwatchState.toSwatch().withUpdatedGauge(gauge)
            }
        }
        viewModelScope.launch {
            val swatch = gaugeStateToSwatch()
            if (swatch == null) {
                _gaugeSuccessfullySaved.emit(false)
                return@launch
            }
            val result = swatchRepository.update(swatch)
            val success = result is DatabaseResult.Success

            if (success) {
                _swatchState.update { SwatchState.from(swatch) }
            }

            _gaugeSuccessfullySaved.emit(success)
        }
    }
}