package net.damschen.swatchit.ui.viewmodels

import android.net.Uri
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import net.damschen.swatchit.R
import net.damschen.swatchit.domain.aggregates.swatch.Count
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Size
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.providers.UUIDProvider
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult
import net.damschen.swatchit.infrastructure.resultWrappers.PhotoResult
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.ui.models.CalculationsState
import net.damschen.swatchit.ui.models.GaugeFormState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.PhotoState
import net.damschen.swatchit.ui.models.SwatchFormStateManager
import net.damschen.swatchit.ui.models.SwatchState
import net.damschen.swatchit.ui.models.ValidatedInput
import net.damschen.swatchit.ui.providers.DateTimeProvider

@HiltViewModel(assistedFactory = EditSwatchViewModel.Factory::class)
class EditSwatchViewModel @AssistedInject constructor(
    @Assisted private val swatchId: Int,
    private val swatchRepository: SwatchRepository,
    dateTimeProvider: DateTimeProvider,
    private val photoStorageService: PhotoStorageService,
    private val uuidProvider: UUIDProvider
) : ViewModel() {
    val formManager = SwatchFormStateManager(dateTimeProvider)
    var swatchHasChanged = MutableStateFlow(false)
    var gaugeHasChanged = MutableStateFlow(false)
    private val _loadState = MutableStateFlow<LoadState>(LoadState.Initial)
    val loadState: StateFlow<LoadState> = _loadState.asStateFlow()

    private var _swatchState = MutableStateFlow<SwatchState?>(null)

    private val _gaugeFormState = MutableStateFlow(
        GaugeFormState.init()
    )
    val gaugeFormState: StateFlow<GaugeFormState> = _gaugeFormState

    private val _photoState = MutableStateFlow(PhotoState(null, true))
    val photoState: StateFlow<PhotoState> = _photoState

    private val _gaugeFormSuccessfullySaved = MutableSharedFlow<Boolean>()
    val gaugeFormSuccessfullySaved: SharedFlow<Boolean> = _gaugeFormSuccessfullySaved.asSharedFlow()

    private val _swatchFormSuccessfullySaved = MutableSharedFlow<Boolean>()
    val swatchFormSuccessfullySaved: SharedFlow<Boolean> =
        _swatchFormSuccessfullySaved.asSharedFlow()

    private val _successfullyDeleted = MutableSharedFlow<Boolean>()
    val successfullyDeleted: SharedFlow<Boolean> = _successfullyDeleted.asSharedFlow()

    private val _calculationsState = MutableStateFlow(
        CalculationsState.init()
    )
    val calculationsState: StateFlow<CalculationsState> = _calculationsState

    init {
        loadSwatch()
        viewModelScope.launch {
            formManager.swatchFormState.collect {
                swatchHasChanged.value = true
            }

        }
        viewModelScope.launch {
            gaugeFormState.collect {
                gaugeHasChanged.value = true
            }
        }
    }

    @AssistedFactory
    interface Factory {
        fun create(swatchId: Int): EditSwatchViewModel
    }

    fun updateFormStates() {
        fun validateGaugeForm() {
            val gaugeState = _gaugeFormState.value
            onNrOfStitchesChange(gaugeState.nrOfStitches.value)
            onNrOfRowsChange(gaugeState.nrOfRows.value)
            onGaugeLengthChange(gaugeState.size.value)
        }
        _swatchState.value?.let { swatch ->
            formManager.reloadSwatchFormState(swatch)
            _gaugeFormState.value = GaugeFormState.fromGauge(swatch.gauge)
            validateGaugeForm()
            swatchHasChanged.value = false
            gaugeHasChanged.value = false
        }
    }

    fun loadSwatch() {
        _loadState.value = LoadState.Loading
        viewModelScope.launch {
            when (val result = swatchRepository.get(swatchId)) {
                is DatabaseResult.Success -> {
                    result.data?.let { swatch ->
                        _swatchState.value = SwatchState.from(swatch)
                        _photoState.value = PhotoState(swatch.photo?.fileName, true)
                        updateFormStates()
                        _loadState.value = LoadState.Success
                    } ?: run {
                        _loadState.value = LoadState.NotFound
                    }
                }

                is DatabaseResult.Error -> {
                    _loadState.value = LoadState.Error(result.exception.message)
                }
            }
        }
    }

    fun saveSwatchForm() {
        viewModelScope.launch {
            val swatchState = _swatchState.value
            if (!formManager.swatchFormIsValid || swatchState == null) {
                _swatchFormSuccessfullySaved.emit(false)
                return@launch
            }

            val swatch = formManager.mergeStateWith(swatchState.toSwatch())

            val result = swatchRepository.update(swatch)
            val success = result is DatabaseResult.Success

            if (success) {
                _swatchState.update { SwatchState.from(swatch) }
                updateFormStates()
            }
            _swatchFormSuccessfullySaved.emit(success)
        }
    }

    fun deleteSwatch() {
        viewModelScope.launch {
            val result = swatchRepository.delete(SwatchId(swatchId))
            val success = result is DatabaseResult.Success
            _successfullyDeleted.emit(success)
        }
    }

    fun saveGaugeForm() {
        fun gaugeStateToSwatch(): Swatch? {
            return _swatchState.value?.let { oldSwatchState ->
                val gauge = _gaugeFormState.value.toGauge()
                oldSwatchState.toSwatch().withUpdatedGauge(gauge)
            }
        }
        viewModelScope.launch {
            if (!_gaugeFormState.value.isValid()) {
                _gaugeFormSuccessfullySaved.emit(false)
                return@launch
            }

            val swatch = gaugeStateToSwatch()
            if (swatch == null) {
                _gaugeFormSuccessfullySaved.emit(false)
                return@launch
            }
            val result = swatchRepository.update(swatch)
            val success = result is DatabaseResult.Success

            if (success) {
                _swatchState.update { SwatchState.from(swatch) }
                updateFormStates()
            }

            _gaugeFormSuccessfullySaved.emit(success)
        }
    }

    fun deleteSwatchPhoto() {
        val swatch = _swatchState.value
        if (swatch == null || swatch.photo == null) return
        val fileDeletionResult = photoStorageService.deleteFromLocalFileSystem(swatch.photo)

        if (fileDeletionResult is PhotoResult.Error) {
            _photoState.update { it.copy(lastUpdateSuccessful = false) }
            return
        }
        viewModelScope.launch {
            val swatchWithoutPhoto = swatch.toSwatch().withoutPhoto()
            val result = swatchRepository.update(swatchWithoutPhoto)
            val success = result is DatabaseResult.Success

            if (success) {
                _swatchState.update { SwatchState.from(swatchWithoutPhoto) }
                _photoState.update {
                    PhotoState(
                        fileName = null,
                        lastUpdateSuccessful = true
                    )
                }
            } else {
                _photoState.update { it.copy(lastUpdateSuccessful = false) }
            }
        }
    }

    fun updateSwatchPhoto(uri: Uri) {
        if (uri == Uri.EMPTY) {
            _photoState.update { it.copy(lastUpdateSuccessful = false) }
            return
        }
        val photo = Photo.create(uuidProvider)
        viewModelScope.launch {
            val fileCreationResult = photoStorageService.copyToLocalFileSystem(uri, photo)

            if (fileCreationResult is PhotoResult.Error) {
                _photoState.update { it.copy(lastUpdateSuccessful = false) }
                return@launch
            }

            val swatchState = _swatchState.value ?: return@launch

            val swatchWithPhoto = swatchState.toSwatch().withUpdatedPhoto(photo)
            val result = swatchRepository.update(swatchWithPhoto)
            val success = result is DatabaseResult.Success

            if (success) {
                if (swatchState.photo != null) photoStorageService.deleteFromLocalFileSystem(
                    swatchState.photo
                )
                _swatchState.update { SwatchState.from(swatchWithPhoto) }
                _photoState.update {
                    PhotoState(
                        fileName = photo.fileName,
                        lastUpdateSuccessful = true
                    )
                }
            } else {
                photoStorageService.deleteFromLocalFileSystem(photo)
                _photoState.update { it.copy(lastUpdateSuccessful = false) }
            }
        }
    }

    fun onNrOfStitchesChange(value: String) {
        val validationResult = GaugeCount.validate(value)

        _gaugeFormState.update { currentState ->
            currentState.copy(
                nrOfStitches = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onNrOfRowsChange(value: String) {
        val validationResult = GaugeCount.validate(value)

        _gaugeFormState.update { currentState ->
            currentState.copy(
                nrOfRows = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onGaugeLengthChange(value: String) {
        val validationResult = GaugeSize.validate(value)
        _gaugeFormState.update { currentState ->
            currentState.copy(
                size = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onCalculationWidthChange(value: String) {
        val validationResult = Size.validate(value)
        _calculationsState.update { currentState ->
            val width = ValidatedInput(value, validationResult.errorMessageId)
            currentState.copy(width = width, nrOfStitches = updateCount(width) { gauge, width ->
                gauge.calculateStitchesFor(width)
            })
        }
    }

    private fun updateCount(
        validatedWidth: ValidatedInput,
        calculateCount: (Gauge, Size) -> Count?
    ): ValidatedInput {
        val gauge = _swatchState.value?.gauge
        if (!validatedWidth.isValid() || gauge == null) return ValidatedInput()

        return Size.create(validatedWidth.value)?.let { width ->
            val result = calculateCount(gauge, (width))
            result?.let { ValidatedInput(it.toString()) } ?: ValidatedInput(
                "",
                R.string.could_not_calculate
            )
        } ?: ValidatedInput()
    }

    fun onCalculationStitchesChange(value: String) {
        val validationResult = Count.validate(value)
        _calculationsState.update { currentState ->
            val stitches = ValidatedInput(value, validationResult.errorMessageId)
            currentState.copy(width = updateSize(stitches) { gauge, stitches ->
                gauge.calculateWidthFor(
                    stitches
                )
            }, nrOfStitches = stitches)
        }
    }

    private fun updateSize(
        validatedStitches: ValidatedInput,
        calculateSize: (Gauge, Count) -> Size
    ): ValidatedInput {
        val gauge = _swatchState.value?.gauge
        if (!validatedStitches.isValid() || gauge == null) return ValidatedInput()

        return Count.create(validatedStitches.value)?.let { stitches ->

            ValidatedInput(calculateSize(gauge, stitches).toString())

        } ?: ValidatedInput()
    }

    fun onCalculationHeightChange(value: String) {
        val validationResult = Size.validate(value)
        _calculationsState.update { currentState ->
            val height = ValidatedInput(value, validationResult.errorMessageId)
            currentState.copy(height = height, nrOfRows = updateCount(height) { gauge, length ->
                gauge.calculateRowsFor(length)
            })
        }
    }

    fun onCalculationRowsChange(value: String) {
        val validationResult = Count.validate(value)
        _calculationsState.update { currentState ->
            val rows = ValidatedInput(value, validationResult.errorMessageId)
            currentState.copy(height = updateSize(rows) { gauge, rows ->
                gauge.calculateHeightFor(
                    rows
                )
            }, nrOfRows = rows)
        }
    }
}