package net.damschen.swatchit.ui.screens

import android.net.Uri
import androidx.activity.compose.BackHandler
import androidx.activity.compose.rememberLauncherForActivityResult
import androidx.activity.result.PickVisualMediaRequest
import androidx.activity.result.contract.ActivityResultContracts
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.FlowRow
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Calculate
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material.icons.filled.Image
import androidx.compose.material.icons.filled.Info
import androidx.compose.material.icons.filled.Menu
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Alignment.Companion.CenterHorizontally
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import coil3.compose.AsyncImage
import net.damschen.swatchit.R
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.ui.elements.CardCaption
import net.damschen.swatchit.ui.elements.ConfirmDeleteDialog
import net.damschen.swatchit.ui.elements.DetailsCard
import net.damschen.swatchit.ui.elements.ErrorDialog
import net.damschen.swatchit.ui.elements.ErrorMessage
import net.damschen.swatchit.ui.elements.LabeledText
import net.damschen.swatchit.ui.elements.ProgressIndicator
import net.damschen.swatchit.ui.elements.ValidatedTextField
import net.damschen.swatchit.ui.models.GaugeFormState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.PhotoState
import net.damschen.swatchit.ui.models.SwatchFormState
import net.damschen.swatchit.ui.models.toFormattedDate
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import java.io.File

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SwatchDetailsScreen(
    viewModel: EditSwatchViewModel,
    onNavigateBack: () -> Unit,
    onEdit: () -> Unit,
    onViewMeasurements: () -> Unit,
    onViewCalculations: () -> Unit
) {
    val swatchState by viewModel.formManager.swatchFormState.collectAsState()
    val gaugeState by viewModel.gaugeFormState.collectAsState()
    val photoState by viewModel.photoState.collectAsState()
    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())
    val loadState by viewModel.loadState.collectAsState()
    var gaugeDialogVisible by remember { mutableStateOf(false) }
    val updateSuccessful by viewModel.gaugeFormSuccessfullySaved.collectAsState(true)
    var showDeleteError by remember { mutableStateOf(false) }

    LaunchedEffect(Unit) {
        viewModel.gaugeFormSuccessfullySaved.collect { success ->
            if (success) {
                gaugeDialogVisible = false
            }
        }
    }

    LaunchedEffect(Unit) {
        viewModel.successfullyDeleted.collect { success ->
            if (success) {
                onNavigateBack()
            } else {
                showDeleteError = true
            }
        }
    }

    BackHandler { onNavigateBack() }

    Scaffold(modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection), topBar = {
        CenterAlignedTopAppBar(
            navigationIcon = {
                IconButton(onClick = {
                    onNavigateBack()
                }, modifier = Modifier.testTag("DismissButton")) {
                    Icon(
                        imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                        contentDescription = stringResource(R.string.back_button_description)
                    )
                }
            },
            actions = {
                IconButton(
                    modifier = Modifier.testTag("EditButton"),
                    onClick = { onEdit() },
                    enabled = swatchState.isValid()
                ) {
                    Icon(
                        imageVector = Icons.Filled.Edit,
                        contentDescription = stringResource(R.string.edit_button_description)
                    )
                }
                OptionsMenu(
                    onDeleteConfirmed = {
                        viewModel.deleteSwatch()
                    },
                    onViewMeasurements = onViewMeasurements,
                    onViewCalculations = onViewCalculations,
                    calculationsEnabled = !gaugeState.isEmpty
                )
            },
            scrollBehavior = scrollBehavior,
            title = {
                Text(
                    swatchState.name.value,
                    maxLines = 1,
                    overflow = TextOverflow.Ellipsis,
                    modifier = Modifier.testTag("Name")
                )
            },
        )
    }) { innerPadding ->
        when (val state = loadState) {
            LoadState.Loading -> {
                ProgressIndicator(innerPadding)
            }

            is LoadState.Error -> {
                ErrorMessage(
                    message = state.message ?: stringResource(R.string.unknown_error),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding,
                    testTag = "LoadStateError"
                )
            }

            LoadState.NotFound -> {
                ErrorMessage(
                    message = stringResource(R.string.swatch_not_found),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding,
                    testTag = "LoadStateNotFound"
                )
            }

            LoadState.Success -> {
                SwatchDetailsColumn(
                    innerPadding = innerPadding,
                    swatchState = swatchState,
                    gaugeState = gaugeState,
                    photoState = photoState,
                    onViewMeasurements = onViewMeasurements,
                    onEditGauge = { gaugeDialogVisible = true },
                    onViewCalculations = onViewCalculations,
                    onPickPhoto = { uri -> viewModel.updateSwatchPhoto(uri) },
                    onDeletePhoto = { viewModel.deleteSwatchPhoto() }
                )
                when {
                    gaugeDialogVisible -> {
                        EditGaugeDialog(gaugeState, viewModel, updateSuccessful, onCancel = {
                            gaugeDialogVisible = false
                            viewModel.updateFormStates()
                        }, onSave = {
                            viewModel.saveGaugeForm()
                        })
                    }
                }
            }

            LoadState.Initial -> {}
        }

        if (showDeleteError) {
            ErrorDialog(
                text = stringResource(R.string.error_while_deleting_swatch), onRetry = {
                    showDeleteError = false
                    viewModel.deleteSwatch()
                }, onCancel = {
                    showDeleteError = false
                }, testTag = "DeleteErrorDialog"
            )
        }
    }
}

@Composable
private fun OptionsMenu(
    onDeleteConfirmed: () -> Unit,
    onViewMeasurements: () -> Unit,
    onViewCalculations: () -> Unit,
    calculationsEnabled: Boolean
) {
    var expanded by remember { mutableStateOf(false) }
    var alertDialogVisible by remember { mutableStateOf(false) }

    IconButton(onClick = { expanded = !expanded }, modifier = Modifier.testTag("OptionsMenu")) {
        Icon(
            imageVector = Icons.Filled.Menu, contentDescription = stringResource(R.string.main_menu)
        )
    }
    DropdownMenu(expanded = expanded, onDismissRequest = { expanded = false }) {
        DropdownMenuItem(
            text = { Text(stringResource(R.string.delete)) },
            modifier = Modifier.testTag("DeleteButton"),
            onClick = {
                alertDialogVisible = true
            },
            leadingIcon = {
                Icon(
                    imageVector = Icons.Filled.Delete,
                    contentDescription = stringResource(R.string.delete_swatch_description)
                )
            })
        DropdownMenuItem(
            text = { Text(stringResource(R.string.view_measurements)) },
            modifier = Modifier.testTag("ViewMeasurementsDropdownItem"),
            onClick = onViewMeasurements,
            leadingIcon = {
                Icon(
                    imageVector = Icons.Filled.Info,
                    contentDescription = stringResource(R.string.view_measurements_description)
                )
            })
        DropdownMenuItem(
            text = { Text(stringResource(R.string.calculations)) },
            modifier = Modifier.testTag("CalculationsDropdownItem"),
            onClick = onViewCalculations,
            enabled = calculationsEnabled,
            leadingIcon = {
                Icon(
                    imageVector = Icons.Filled.Calculate,
                    contentDescription = stringResource(R.string.view_calculations_description)
                )
            })

        when {
            alertDialogVisible -> {
                val text =
                    stringResource(R.string.clicking_confirm_will_permanently_delete_the_swatch)
                ConfirmDeleteDialog(text, onDismiss = {
                    alertDialogVisible = false
                    expanded = false
                }, onConfirmation = {
                    alertDialogVisible = false
                    expanded = false
                    onDeleteConfirmed()
                })
            }
        }
    }
}

@Composable
private fun SwatchDetailsColumn(
    innerPadding: PaddingValues,
    swatchState: SwatchFormState,
    gaugeState: GaugeFormState,
    photoState: PhotoState,
    onViewMeasurements: () -> Unit,
    onEditGauge: () -> Unit,
    onViewCalculations: () -> Unit,
    onPickPhoto: (uri: Uri) -> Unit,
    onDeletePhoto: () -> Unit
) {
    Column(
        modifier = Modifier
            .padding(innerPadding)
            .verticalScroll(rememberScrollState())
    ) {
        DetailsCard {
            CardCaption(
                text = stringResource(R.string.yarn)
            )
            Row(Modifier.fillMaxWidth()) {
                LabeledText(
                    value = swatchState.yarnManufacturer.value,
                    labelText = stringResource(id = R.string.yarn_manufacturer),
                    testTag = "YarnManufacturer",
                    modifier = Modifier.weight(2f, true)
                )

                LabeledText(
                    value = swatchState.yarnName.value,
                    labelText = stringResource(id = R.string.yarn_name),
                    testTag = "YarnName",
                    modifier = Modifier.weight(2f, true)
                )
            }
        }
        DetailsCard {
            CardCaption(stringResource(R.string.setup))
            Row(Modifier.fillMaxWidth()) {
                LabeledText(
                    value = swatchState.needleSize.displayValue,
                    labelText = stringResource(R.string.needle_size),
                    testTag = "NeedleSize",
                    modifier = Modifier.weight(2f, true)
                )

                LabeledText(
                    value = swatchState.pattern.value,
                    labelText = stringResource(id = R.string.pattern),
                    testTag = "Pattern",
                    modifier = Modifier.weight(2f, true)
                )
            }
        }
        DetailsCard {
            Row {
                CardCaption(
                    text = stringResource(R.string.gauge)
                )
                Spacer(Modifier.weight(1f))
                IconButton(onClick = onEditGauge, modifier = Modifier.testTag("EditGaugeButton")) {
                    Icon(
                        imageVector = Icons.Filled.Edit,
                        contentDescription = stringResource(R.string.enter_edit_gauge)
                    )
                }
            }
            if (!gaugeState.isEmpty) {
                GaugeInformation(gaugeState)
            }

            FlowRow {
                Button(
                    onClick = onViewCalculations, Modifier
                        .padding(8.dp)
                        .testTag("CalculationsButton")
                        .weight(1f), enabled = !gaugeState.isEmpty
                ) {
                    Text(
                        stringResource(R.string.calculations)
                    )
                }
                Button(
                    onClick = onViewMeasurements, Modifier
                        .padding(8.dp)
                        .testTag("ViewMeasurementsButton")
                        .weight(1f)
                ) {
                    Text(
                        stringResource(R.string.view_measurements)
                    )
                }
            }
        }

        DetailsCard {
            LabeledText(
                value = swatchState.notes.value,
                labelText = stringResource(id = R.string.notes),
                testTag = "Notes",
                softWrap = true
            )
        }
        DetailsCard {
            LabeledText(
                value = swatchState.createdAt.toFormattedDate(),
                labelText = stringResource(R.string.created_at),
                testTag = "CreatedAt"
            )
        }
        PhotoPickerCard(photoState, onDeletePhoto, onPickPhoto)
    }
}

@Composable
private fun PhotoPickerCard(
    photoState: PhotoState,
    onDeletePhoto: () -> Unit,
    onPickPhoto: (Uri) -> Unit
) {
    val context = LocalContext.current
    DetailsCard {
        Column {
            Text(
                text = stringResource(R.string.photo),
                style = MaterialTheme.typography.labelLarge,
                modifier = Modifier
                    .padding(24.dp, 12.dp, 24.dp, 0.dp)
                    .testTag("SwatchPhotoCaption")
            )

            val file = photoState.fileName?.let {
                File(context.filesDir, "${PhotoStorageService.IMAGES_DIR_NAME}/$it")
            }

            if (file != null && !file.exists())
                Text(
                    text = stringResource(R.string.bitmap_not_found_message),
                    modifier = Modifier
                        .padding(24.dp)
                        .testTag("BitmapNotFoundError"),
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.error
                )
            Box(
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(24.dp, 12.dp),
                contentAlignment = Alignment.Center
            ) {
                AsyncImage(
                    model = file,
                    contentDescription = stringResource(R.string.photo_of_the_swatch),
                    modifier = Modifier
                        .clip(RoundedCornerShape(12.dp))
                        .testTag("SwatchBitmap"),
                    contentScale = ContentScale.FillWidth
                )
            }

            if (!photoState.lastUpdateSuccessful) {
                Text(
                    text = stringResource(R.string.error_updating_photo),
                    modifier = Modifier
                        .padding(24.dp)
                        .testTag("PhotoUpdateError"),
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.error
                )
            }
            FlowRow {
                DeletePhotoButton(
                    enabled = photoState.fileName != null,
                    onDeletePhoto = onDeletePhoto
                )
                Spacer(Modifier.weight(1f))
                PhotoPickerButton(onPickPhoto = onPickPhoto)
            }
        }
    }
}

@Composable
private fun PhotoPickerButton(onPickPhoto: (uri: Uri) -> Unit) {
    val pickMedia =
        rememberLauncherForActivityResult(ActivityResultContracts.PickVisualMedia()) { uri ->
            if (uri != null) {
                onPickPhoto(uri)
            }
        }
    Button(
        onClick = { pickMedia.launch(PickVisualMediaRequest(ActivityResultContracts.PickVisualMedia.ImageOnly)) },
        modifier = Modifier
            .padding(24.dp, 0.dp, 24.dp, 12.dp)
            .testTag("PhotoPickerButton")
    ) {
        Icon(
            imageVector = Icons.Filled.Image,
            contentDescription = stringResource(R.string.pick_a_photo)
        )
        Text(stringResource(R.string.pick))
    }
}

@Composable
private fun DeletePhotoButton(enabled: Boolean, onDeletePhoto: () -> Unit) {
    var alertDialogVisible by remember { mutableStateOf(false) }

    Button(
        onClick = { alertDialogVisible = true },
        modifier = Modifier
            .padding(24.dp, 0.dp, 24.dp, 12.dp)
            .testTag("DeletePhotoButton"),
        enabled = enabled
    ) {
        Icon(
            imageVector = Icons.Filled.Delete,
            contentDescription = "Delete the photo"
        )
        Text(stringResource(R.string.delete))
    }
    when {
        alertDialogVisible -> {
            val text =
                stringResource(R.string.clicking_confirm_will_permanently_delete_the_photo)
            ConfirmDeleteDialog(text, onDismiss = {
                alertDialogVisible = false
            }, onConfirmation = {
                alertDialogVisible = false
                onDeletePhoto()
            })
        }
    }
}

@Composable
private fun GaugeInformation(gaugeState: GaugeFormState) {
    Row(Modifier.fillMaxWidth()) {
        LabeledText(
            value = gaugeState.nrOfStitches.value,
            labelText = stringResource(id = R.string.nr_of_stitches),
            testTag = "NrOfStitches",
            modifier = Modifier.weight(2f, true)
        )
        LabeledText(
            value = gaugeState.nrOfRows.value,
            labelText = stringResource(id = R.string.nr_of_rows),
            testTag = "NrOfRows",
            modifier = Modifier.weight(2f, true)
        )
    }

    LabeledText(
        value = "${gaugeState.size.value} cm",
        labelText = stringResource(id = R.string.Per),
        testTag = "GaugeLength"
    )
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun EditGaugeDialog(
    state: GaugeFormState,
    viewModel: EditSwatchViewModel,
    saveSuccessful: Boolean,
    onCancel: () -> Unit,
    onSave: () -> Unit
) {
    val hasChanged by viewModel.gaugeHasChanged.collectAsState()
    Dialog(onDismissRequest = { onCancel() }) {
        Card(
            modifier = Modifier
                .fillMaxWidth()
                .height(450.dp)
                .padding(16.dp)
                .testTag("EditGaugeDialog"),
            shape = RoundedCornerShape(16.dp),
        ) {
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState()),
                verticalArrangement = Arrangement.SpaceEvenly,
                horizontalAlignment = CenterHorizontally,
            ) {
                Text(
                    text = stringResource(R.string.edit_gauge),
                    modifier = Modifier.padding(16.dp),
                    style = MaterialTheme.typography.headlineSmall
                )
                if (!saveSuccessful) {
                    Card(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(horizontal = 24.dp),
                        colors = CardDefaults.cardColors(
                            containerColor = MaterialTheme.colorScheme.errorContainer
                        )
                    ) {
                        Text(
                            text = stringResource(R.string.error_while_updating_swatch),
                            modifier = Modifier
                                .padding(12.dp)
                                .testTag("GaugeSaveError"),
                            style = MaterialTheme.typography.bodySmall,
                            color = MaterialTheme.colorScheme.onErrorContainer
                        )
                    }
                }

                ValidatedTextField(
                    validatedInput = state.nrOfStitches,
                    onValueChange = {
                        viewModel.onNrOfStitchesChange(it)
                    },
                    labelText = stringResource(id = R.string.nr_of_stitches),
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    testTag = "NrOfStitchesInput"
                )

                ValidatedTextField(
                    validatedInput = state.nrOfRows,
                    onValueChange = {
                        viewModel.onNrOfRowsChange(it)
                    },
                    labelText = stringResource(id = R.string.nr_of_rows),
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    testTag = "NrOfRowsInput"
                )

                ValidatedTextField(
                    validatedInput = state.size,
                    onValueChange = {
                        viewModel.onGaugeLengthChange(it)
                    },
                    labelText = stringResource(id = R.string.Per_cm),
                    keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                    testTag = "GaugeLengthInput"
                )
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.Center,
                ) {
                    TextButton(
                        onClick = { onCancel() },
                        modifier = Modifier
                            .padding(8.dp)
                            .testTag("CancelEditGaugeButton"),
                    ) {
                        Text(stringResource(R.string.cancel))
                    }
                    TextButton(
                        enabled = state.isValid() && hasChanged, onClick = {
                            onSave()
                        }, modifier = Modifier
                            .padding(8.dp)
                            .testTag("SaveGaugeButton")
                    ) {
                        Text(stringResource(R.string.save))
                    }
                }
            }
        }
    }
}