package net.damschen.swatchit.ui.models

import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.update
import net.damschen.swatchit.domain.aggregates.swatch.Name
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import net.damschen.swatchit.ui.providers.DateTimeProvider

class SwatchFormStateManager(dateTimeProvider: DateTimeProvider) {
    private val defaultCreatedAt: Long = dateTimeProvider.now().time
    private val _swatchFormState = MutableStateFlow(
        SwatchFormState.init(defaultCreatedAt)
    )
    val swatchFormState: StateFlow<SwatchFormState> = _swatchFormState

    val swatchFormIsValid get() = _swatchFormState.value.isValid()
    fun resetFormState() {
        _swatchFormState.value = SwatchFormState.init(defaultCreatedAt)
    }

    fun mergeStateWith(swatch: Swatch) = swatchFormState.value.mergeWith(swatch)

    fun reloadSwatchFormState(swatchState: SwatchState) {
        _swatchFormState.value = SwatchFormState.fromSwatchState(swatchState)
    }

    fun onNameChange(value: String) {
        val validationResult = Name.validate(value)
        _swatchFormState.update { currentState ->
            currentState.copy(
                name = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onYarnManufacturerChange(value: String) {
        val validationResult = Name.validate(value)
        _swatchFormState.update { currentState ->
            currentState.copy(
                yarnManufacturer = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onYarnNameChange(value: String) {
        val validationResult = Name.validate(value)
        _swatchFormState.update { currentState ->
            currentState.copy(
                yarnName = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onNeedleSizeChange(value: KnittingNeedleSize) {
        _swatchFormState.update { currentState ->
            currentState.copy(needleSize = value)
        }
    }

    fun onPatternChange(value: String) {
        val validationResult = Pattern.validate(value)
        _swatchFormState.update { currentState ->
            currentState.copy(
                pattern = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onNotesChange(value: String) {
        val validationResult = Notes.validate(value)
        _swatchFormState.update { currentState ->
            currentState.copy(
                notes = ValidatedInput(value, validationResult.errorMessageId)
            )
        }
    }

    fun onCreatedAtChange(value: Long) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                createdAt = value
            )
        }
    }
}