package net.damschen.swatchit.ui.models

import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Name
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import net.damschen.swatchit.ui.enums.toDataEnum
import net.damschen.swatchit.ui.enums.toUiEnum
import java.text.SimpleDateFormat
import java.util.Locale

data class SwatchFormState(
    val yarnManufacturer: ValidatedInput,
    val yarnName: ValidatedInput,
    val pattern: ValidatedInput,
    val needleSize: KnittingNeedleSize,
    val notes: ValidatedInput,
    val createdAt: Long,
    val name: ValidatedInput,
    val id: Int? = null
) {
    companion object {
        fun fromSwatchState(swatch: SwatchState): SwatchFormState {
            requireNotNull(swatch.id)
            return SwatchFormState(
                yarnManufacturer = ValidatedInput(swatch.yarn?.manufacturer?.value ?: ""),
                yarnName = ValidatedInput(swatch.yarn?.name?.value ?: ""),
                pattern = ValidatedInput(swatch.pattern?.value ?: ""),
                needleSize = swatch.needleSize.toUiEnum(),
                notes = ValidatedInput(swatch.notes?.value ?: ""),
                createdAt = swatch.createdAt.value,
                id = swatch.id.value,
                name = ValidatedInput(swatch.name?.value ?: "")
            )
        }

        fun init(createdAt: Long): SwatchFormState {
            val defaultNeedleSize: KnittingNeedleSize = KnittingNeedleSize.SIZE_2_5
            return SwatchFormState(
                yarnManufacturer = ValidatedInput(),
                yarnName = ValidatedInput(),
                pattern = ValidatedInput(),
                needleSize = defaultNeedleSize,
                notes = ValidatedInput(),
                createdAt = createdAt,
                name = ValidatedInput(),
                id = null
            )
        }
    }

    fun isValid(): Boolean =
        yarnManufacturer.isValid() && yarnName.isValid() && pattern.isValid() && notes.isValid() && name.isValid()

    fun toSwatch(): Swatch {
        return Swatch.create(
            needleSize = needleSize.toDataEnum(),
            pattern = Pattern.create(pattern.value),
            yarn = Yarn.create(
                Name.create(yarnName.value), Name.create(yarnManufacturer.value)
            ),
            notes = Notes.create(notes.value),
            createdAt = EpochMillis(createdAt),
            id = id?.let { SwatchId(id) },
            name = Name.create(name.value)
        )
    }

    fun mergeWith(source: Swatch): Swatch {
        return toSwatch().withNewMeasurements(source.measurements).let { swatch ->
            val merged = source.gauge?.let { swatch.withUpdatedGauge(it) } ?: swatch
            source.photo?.let { merged.withUpdatedPhoto(it) } ?: merged
        }
    }
}

fun Long.toFormattedDate(): String {
    val formatter = SimpleDateFormat("dd/MM/yyyy", Locale.getDefault())
    return formatter.format(this)
}