package net.damschen.swatchit.ui.elements

import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.unit.dp
import net.damschen.swatchit.ui.models.ValidatedInput

@Composable
fun ValidatedTextField(
    validatedInput: ValidatedInput,
    onValueChange: (String) -> Unit,
    labelText: String,
    keyboardOptions: KeyboardOptions = KeyboardOptions.Default,
    testTag: String = "",
    singleLine: Boolean = true
) {
    var hasBeenFocused by remember { mutableStateOf(false) }
    val focusManager = LocalFocusManager.current

    OutlinedTextField(
        value = validatedInput.value,
        modifier = Modifier
            .fillMaxWidth()
            .padding(24.dp, 6.dp)
            .testTag(testTag)
            .onFocusChanged {
                if (it.isFocused) hasBeenFocused = true
            },
        onValueChange = onValueChange,
        label = { Text(text = labelText) },
        isError = !validatedInput.isValid() && hasBeenFocused,
        keyboardOptions = keyboardOptions.merge(KeyboardOptions(imeAction = ImeAction.Next)),
        singleLine = singleLine,
        supportingText = {
            if (validatedInput.errorMessageId != null && hasBeenFocused) {
                Text(
                    text = stringResource(validatedInput.errorMessageId),
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.onErrorContainer
                )
            }
        },
        keyboardActions = KeyboardActions(onNext = { focusManager.moveFocus(FocusDirection.Down) }),
    )
}
