package net.damschen.swatchit.ui.elements

import androidx.compose.foundation.gestures.awaitEachGesture
import androidx.compose.foundation.gestures.awaitFirstDown
import androidx.compose.foundation.gestures.waitForUpOrCancellation
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material.icons.filled.DateRange
import androidx.compose.material3.DatePicker
import androidx.compose.material3.DatePickerDialog
import androidx.compose.material3.DisplayMode
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ExposedDropdownMenuAnchorType
import androidx.compose.material3.ExposedDropdownMenuBox
import androidx.compose.material3.Icon
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.rememberDatePickerState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.PointerEventPass
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import net.damschen.swatchit.R
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import net.damschen.swatchit.ui.models.SwatchFormState
import net.damschen.swatchit.ui.models.SwatchFormStateManager
import net.damschen.swatchit.ui.models.toFormattedDate

@Composable
@OptIn(ExperimentalMaterial3Api::class)
fun SwatchPropertiesColumn(
    innerPadding: PaddingValues, manager: SwatchFormStateManager, swatchState: SwatchFormState
) {
    var needleSizeExpanded by remember { mutableStateOf(false) }

    Column(
        modifier = Modifier
            .fillMaxSize()
            .padding(innerPadding)
            .verticalScroll(rememberScrollState())
            .imePadding() // to allow for scrolling when keyboard is up
    ) {
        ValidatedTextField(
            validatedInput = swatchState.name,
            onValueChange = {
                manager.onNameChange(it)
            },
            labelText = stringResource(id = R.string.swatch_name),
            testTag = "Name"
        )
        ValidatedTextField(
            validatedInput = swatchState.yarnManufacturer,
            onValueChange = {
                manager.onYarnManufacturerChange(it)
            },
            labelText = stringResource(id = R.string.yarn_manufacturer),
            testTag = "YarnManufacturer"
        )

        ValidatedTextField(
            validatedInput = swatchState.yarnName,
            onValueChange = {
                manager.onYarnNameChange(it)
            },
            labelText = stringResource(id = R.string.yarn_name),
            testTag = "YarnName"
        )

        ExposedDropdownMenuBox(
            expanded = needleSizeExpanded,
            onExpandedChange = { needleSizeExpanded = !needleSizeExpanded },
            modifier = Modifier
                .fillMaxWidth()
                .padding(24.dp, 6.dp)
        ) {
            OutlinedTextField(
                value = swatchState.needleSize.displayValue,
                onValueChange = {},
                label = { Text(stringResource(R.string.needle_size)) },
                readOnly = true,
                trailingIcon = {
                    Icon(
                        Icons.Default.ArrowDropDown,
                        contentDescription = stringResource(R.string.drop_down_description)
                    )
                },
                modifier = Modifier
                    .fillMaxWidth()
                    .menuAnchor(
                        type = ExposedDropdownMenuAnchorType.PrimaryEditable,
                        enabled = true
                    )
                    .testTag("NeedleSize")
            )
            ExposedDropdownMenu(
                expanded = needleSizeExpanded,
                onDismissRequest = { needleSizeExpanded = false }) {
                KnittingNeedleSize.entries.forEach { size ->
                    DropdownMenuItem(text = { Text(size.displayValue) }, onClick = {
                        manager.onNeedleSizeChange(size)
                        needleSizeExpanded = false
                    })
                }
            }
        }

        ValidatedTextField(
            validatedInput = swatchState.pattern,
            onValueChange = {
                manager.onPatternChange(it)
            },
            labelText = stringResource(id = R.string.pattern),
            testTag = "Pattern"
        )

        ValidatedTextField(
            validatedInput = swatchState.notes,
            onValueChange = {
                manager.onNotesChange(it)
            },
            labelText = stringResource(id = R.string.notes),
            testTag = "Notes",
            singleLine = false
        )
        DatePickerFieldToModal(swatchState, onDateSelected = { manager.onCreatedAtChange(it) })
    }
}

@Composable
fun DatePickerFieldToModal(state: SwatchFormState, onDateSelected: (Long) -> Unit) {
    var showModal by remember { mutableStateOf(false) }

    OutlinedTextField(
        value = state.createdAt.toFormattedDate(),
        onValueChange = { },
        label = { Text(text = stringResource(R.string.created_at)) },
        placeholder = { Text("DD/MM/YYYY") },
        trailingIcon = {
            Icon(
                Icons.Default.DateRange,
                contentDescription = stringResource(R.string.date_picker_description)
            )
        }, readOnly = true,
        modifier = Modifier
            .testTag("CreatedAt")
            .fillMaxWidth()
            .padding(24.dp, 6.dp)
            .pointerInput(state.createdAt) {
                awaitEachGesture {
                    awaitFirstDown(pass = PointerEventPass.Initial)
                    val upEvent = waitForUpOrCancellation(pass = PointerEventPass.Initial)
                    if (upEvent != null) {
                        showModal = true
                    }
                }
            })
    if (showModal) {
        DatePickerModal(onDateSelected = onDateSelected, onDismiss = { showModal = false })
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun DatePickerModal(
    onDateSelected: (Long) -> Unit, onDismiss: () -> Unit
) {
    val datePickerState = rememberDatePickerState(initialDisplayMode = DisplayMode.Picker)

    DatePickerDialog(
        modifier = Modifier.testTag("DatePickerDialog"),
        onDismissRequest = onDismiss,
        confirmButton = {
            TextButton(enabled = datePickerState.selectedDateMillis != null, onClick = {
                onDateSelected(datePickerState.selectedDateMillis!!)
                onDismiss()
            }, modifier = Modifier.testTag("DatePickerOkButton")) {
                Text(stringResource(R.string.ok))
            }
        },
        dismissButton = {
            TextButton(onClick = onDismiss) {
                Text(stringResource(R.string.cancel))
            }
        }) {
        DatePicker(state = datePickerState)
    }
}