package net.damschen.swatchit.domain.aggregates.swatch

import net.damschen.swatchit.R
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.domain.resultWrappers.ValidationResult

data class Size(val value: Double) {
    val unit: LengthUnit = LengthUnit.CM

    init {
        require(validate(value).isValid) { "Size invalid" }
    }

    companion object {
        fun validate(value: Double): ValidationResult {
            return when {
                value < MIN -> ValidationResult.error(R.string.must_be_greater_than_zero)
                else -> ValidationResult.success()
            }
        }

        fun validate(value: String): ValidationResult {
            val trimmed = value.trim()
            when {
                trimmed.isBlank() -> return ValidationResult.success()
                trimmed.substringAfterLast(
                    '.',
                    ""
                ).length > MAX_DECIMAL_NUMBERS -> return ValidationResult.error(R.string.max_1_decimal)

                trimmed.length >= MAX_CHARS -> return ValidationResult.error(R.string.number_too_big)
            }
            val gaugeSize =
                trimmed.toDoubleOrNull()
                    ?: return ValidationResult.error(R.string.invalid_number)
            return validate(gaugeSize)
        }

        fun create(value: String): Size? {
            val trimmed = value.trim()
            require(validate(trimmed).isValid) { "Size invalid" }
            val numericValue = trimmed.toDoubleOrNull()
            return numericValue?.let { Size(it) }
        }
    }

    override fun toString(): String = value.toPlainStringWithoutTrailingZeros()
}

private const val MIN = 0.1
private const val MAX_DECIMAL_NUMBERS = 1
private const val MAX_CHARS = Double.MAX_VALUE.toString().length