package net.damschen.swatchit

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.animation.EnterTransition
import androidx.compose.animation.ExitTransition
import androidx.compose.animation.togetherWith
import androidx.compose.runtime.Composable
import androidx.hilt.lifecycle.viewmodel.compose.hiltViewModel
import androidx.lifecycle.viewmodel.navigation3.rememberViewModelStoreNavEntryDecorator
import androidx.navigation3.runtime.NavKey
import androidx.navigation3.runtime.entryProvider
import androidx.navigation3.runtime.rememberNavBackStack
import androidx.navigation3.runtime.rememberSaveableStateHolderNavEntryDecorator
import androidx.navigation3.ui.NavDisplay
import dagger.hilt.android.AndroidEntryPoint
import kotlinx.serialization.Serializable
import net.damschen.swatchit.ui.screens.AddSwatchScreen
import net.damschen.swatchit.ui.screens.EditSwatchScreen
import net.damschen.swatchit.ui.screens.GaugeCalculationsScreen
import net.damschen.swatchit.ui.screens.MeasurementsScreen
import net.damschen.swatchit.ui.screens.SwatchDetailsScreen
import net.damschen.swatchit.ui.screens.SwatchListScreen
import net.damschen.swatchit.ui.theme.AppTheme
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel

@AndroidEntryPoint
class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            AppTheme {
                SwatchItApp()
            }
        }
    }
}

@Composable
fun SwatchItApp() {

    val backStack = rememberNavBackStack(List)

    val entryProvider = entryProvider {
        entry<List> {
            SwatchListScreen(onAddSwatchClicked = {
                backStack.add(Add)
            }, onSwatchDetailsClicked = { swatchId ->
                backStack.add(SwatchFlow(swatchId))
            })
        }
        entry<Add> {
            AddSwatchScreen(onNavigateBack = {
                backStack.removeLastOrNull()
            })
        }
        entry<SwatchFlow> { key ->
            val editSwatchVm = hiltViewModel<EditSwatchViewModel, EditSwatchViewModel.Factory>(
                creationCallback = { factory ->
                    factory.create(key.id)
                }
            )

            val innerBackStack = rememberNavBackStack(Detail)

            val innerEntryProvider = entryProvider {
                entry<Detail> {
                    SwatchDetailsScreen(
                        viewModel = editSwatchVm,
                        onEdit = { innerBackStack.add(Edit) },
                        onViewMeasurements = { innerBackStack.add(ViewMeasurements) },
                        onViewCalculations = { innerBackStack.add(Calculations) },
                        onNavigateBack = {
                            backStack.removeLastOrNull()
                        }
                    )
                }
                entry<Edit> {
                    EditSwatchScreen(
                        viewModel = editSwatchVm,
                        onNavigateBack = {
                            editSwatchVm.updateFormStates()
                            innerBackStack.removeLastOrNull()
                        }
                    )
                }
                entry<ViewMeasurements> {
                    MeasurementsScreen(
                        itemId = key.id,
                        onNavigateBack = {
                            editSwatchVm.loadSwatch()
                            innerBackStack.removeLastOrNull()
                        }
                    )
                }
                entry<Calculations> {
                    GaugeCalculationsScreen(
                        viewModel = editSwatchVm,
                        onNavigateBack = { innerBackStack.removeLastOrNull() }
                    )
                }
            }

            NavDisplay(
                entryDecorators = listOf(
                    rememberSaveableStateHolderNavEntryDecorator(),
                    rememberViewModelStoreNavEntryDecorator()
                ),
                onBack = { innerBackStack.removeLastOrNull() },
                transitionSpec = { EnterTransition.None togetherWith ExitTransition.None },
                popTransitionSpec = { EnterTransition.None togetherWith ExitTransition.None },
                backStack = innerBackStack,
                entryProvider = innerEntryProvider
            )
        }
    }

    NavDisplay(
        entryDecorators = listOf(
            rememberSaveableStateHolderNavEntryDecorator(),
            rememberViewModelStoreNavEntryDecorator()
        ),
        onBack = { backStack.removeLastOrNull() },
        transitionSpec = { EnterTransition.None togetherWith ExitTransition.None },
        popTransitionSpec = { EnterTransition.None togetherWith ExitTransition.None },
        backStack = backStack,
        entryProvider = entryProvider
    )
}

@Serializable
data class SwatchFlow(val id: Int) : NavKey

@Serializable
object Edit : NavKey

@Serializable
object ViewMeasurements : NavKey

@Serializable
object Detail : NavKey

@Serializable
object Add : NavKey

@Serializable
object List : NavKey

@Serializable
object Calculations : NavKey
