package net.damschen.swatchit.test.ui.screens

import android.content.Context
import androidx.compose.ui.test.assertCountEquals
import androidx.compose.ui.test.assertIsNotDisplayed
import androidx.compose.ui.test.assertTextContains
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onChildren
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.test.core.app.ApplicationProvider
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.shared.testhelpers.createTestImageFileInFileSystem
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.enums.toUiEnum
import net.damschen.swatchit.ui.screens.SwatchListScreen
import net.damschen.swatchit.ui.viewmodels.SwatchListViewModel
import org.junit.Assert.assertEquals
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog

@RunWith(RobolectricTestRunner::class)
class SwatchListScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Test
    fun swatchEntryClicked_callsOnSwatchDetailsClicked() = runTest {
        val repo = FakeRepo()
        val viewModel = SwatchListViewModel(repo)
        var actualSwatchId = -1
        composeTestRule.setContent {
            SwatchListScreen(
                viewModel,
                onAddSwatchClicked = { },
                onSwatchDetailsClicked = { swatchId -> swatchId.also { actualSwatchId = it } })
        }
        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)
        columnItems[0].performClick()

        assertEquals(1, actualSwatchId)
    }

    @Test
    fun swatchListItem_swatchInDatabase_showsSwatchDetailInfo() = runTest {
        val repository = FakeRepo()
        val swatch = repository.swatchToReturn()!!
        val viewModel = SwatchListViewModel(repository)

        composeTestRule.setContent {
            SwatchListScreen(viewModel, onAddSwatchClicked = { }, onSwatchDetailsClicked = {})
        }

        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)
        columnItems[0].assertTextContains(swatch.name!!.value, true)
        columnItems[0].assertTextContains(swatch.yarn!!.name!!.value, true)
        columnItems[0].assertTextContains(swatch.yarn.manufacturer!!.value, true)
        columnItems[0].assertTextContains(swatch.gauge!!.nrOfRows.toString(), true)
        columnItems[0].assertTextContains(swatch.gauge!!.nrOfStitches.toString(), true)
        columnItems[0].assertTextContains(swatch.gauge!!.size.toString(), true)
        columnItems[0].assertTextContains(swatch.needleSize.toUiEnum().displayValue, true)
    }

    @Test
    fun swatchListItem_swatchWithPhotoInDatabase_showsSwatchPhoto() = runTest {
        val context: Context = ApplicationProvider.getApplicationContext()
        createTestImageFileInFileSystem(context)
        val repository = FakeRepo()
        val swatch =
            repository.swatchToReturn()!!.withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID))
        repository.swatchToReturn = { swatch }
        val viewModel = SwatchListViewModel(repository)

        composeTestRule.setContent {
            SwatchListScreen(viewModel, onAddSwatchClicked = { }, onSwatchDetailsClicked = {})
        }

        composeTestRule.onNodeWithTag("Photo_${swatch.id!!.value}", useUnmergedTree = true)
            .assertExists()
    }

    @Test
    fun swatchListItem_swatchWithoutPhotoInDatabase_doesNotShowsSwatchPhoto() = runTest {
        val repository = FakeRepo()
        val swatch = repository.swatchToReturn()!!
        val viewModel = SwatchListViewModel(repository)

        composeTestRule.setContent {
            SwatchListScreen(viewModel, onAddSwatchClicked = { }, onSwatchDetailsClicked = {})
        }

        composeTestRule.onNodeWithTag("Photo_${swatch.id!!.value}", useUnmergedTree = true)
            .assertDoesNotExist()
    }

    @Test
    fun swatchListItem_swatchWithPhotoThatIsNotInFileSystem_doesNotShowsSwatchPhoto() = runTest {
        val repository = FakeRepo()
        val swatch =
            repository.swatchToReturn()!!.withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID))
        repository.swatchToReturn = { swatch }
        val viewModel = SwatchListViewModel(repository)
        composeTestRule.setContent {
            SwatchListScreen(viewModel, onAddSwatchClicked = { }, onSwatchDetailsClicked = {})
        }

        composeTestRule.onNodeWithTag("Photo_${swatch.id!!.value}", useUnmergedTree = true)
            .performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun swatchListItem_swatchWithoutGauge_showsNoGaugeInfo() {
        val repository = FakeRepo()
        repository.swatchToReturn = { defaultSwatchWithoutGauge }
        val viewModel = SwatchListViewModel(repository)

        composeTestRule.setContent {
            SwatchListScreen(viewModel, onAddSwatchClicked = { }, onSwatchDetailsClicked = {})
        }

        val columnItems = composeTestRule.onNodeWithTag("Column").onChildren()
        columnItems.assertCountEquals(1)
        columnItems[0].assertTextContains("No gauge info", true)
    }
}

private val defaultSwatchWithoutGauge = Swatch.create(
    needleSize = KnittingNeedleSize.SIZE_10_0,
    pattern = null,
    yarn = null,
    notes = null,
    createdAt = EpochMillis(0),
    id = SwatchId(1),
    name = null
)