package net.damschen.swatchit.test.ui.screens

import androidx.compose.ui.test.assert
import androidx.compose.ui.test.assertIsDisplayed
import androidx.compose.ui.test.assertIsEnabled
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.hasText
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.compose.ui.test.performTextInput
import androidx.compose.ui.test.performTextReplacement
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.test.testHelpers.FakePhotoStorageService
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.screens.SwatchDetailsScreen
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog

@RunWith(RobolectricTestRunner::class)
class SwatchDetailsScreenUnitTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    private lateinit var viewModel: EditSwatchViewModel
    private var itemId: Int = 1
    private lateinit var defaultSwatch: Swatch
    private var repo = FakeRepo()

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initViewModel() {
        repo = FakeRepo()
        defaultSwatch = repo.swatchToReturn()!!
        itemId = repo.defaultId
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
    }

    @Test
    fun backButtonClicked_callsOnDismiss() {
        var dismissed = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { dismissed = true },
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(dismissButtonTag).performClick()

        assertTrue(dismissed)
    }

    @Test
    fun editButtonClicked_callsOnEdit() {
        var editClicked = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { editClicked = true },
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(editButtonTag).performClick()

        assertTrue(editClicked)
    }

    @Test
    fun viewMeasurementsDropDownItemClicked_callsOnViewMeasurements() {
        var viewMeasurementsClicked = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = { viewMeasurementsClicked = true },
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(viewMeasurementsDropDownItemTag).performClick()

        assertTrue(viewMeasurementsClicked)
    }

    @Test
    fun viewMeasurementsButtonClicked_callsOnViewMeasurements() {
        var viewMeasurementsClicked = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = { viewMeasurementsClicked = true },
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("ViewMeasurementsButton").performScrollTo().performClick()

        assertTrue(viewMeasurementsClicked)
    }

    @Test
    fun calculationsDropDownItemClicked_callsOnViewCalculations() {
        var viewCalculationsClicked = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = { viewCalculationsClicked = true })
        }
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag("CalculationsDropdownItem").performClick()

        assertTrue(viewCalculationsClicked)
    }

    @Test
    fun calculationsDropDownItemClicked_noGaugeSet_calculationsDropdownItemDisabled() {
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag("CalculationsDropdownItem").assertIsNotEnabled()
    }

    @Test
    fun calculationsButtonClicked_callsOnViewCalculations() {
        var viewCalculationsClicked = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = { viewCalculationsClicked = true })
        }
        composeTestRule.onNodeWithTag("CalculationsButton").performScrollTo().performClick()

        assertTrue(viewCalculationsClicked)
    }

    @Test
    fun calculationsButtonClicked_noGaugeSet_buttonDisabled() {
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("CalculationsButton").assertIsNotEnabled()
    }

    @Test
    fun swatchInDatabase_showsName() {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(nameValueTag).assert(hasText(defaultSwatch.name!!.value))
    }

    @Test
    fun swatchWithGauge_showsGaugeDetails() {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(nrOfStitchesValueTag).assertExists()
        composeTestRule.onNodeWithTag(nrOfRowsValueTag).assertExists()
        composeTestRule.onNodeWithTag(gaugeLengthValueTag).assertExists()
    }

    @Test
    fun swatchWithoutGauge_doesNotShowGaugeDetails() {
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { },
                onEdit = { },
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(nrOfStitchesValueTag).assertDoesNotExist()
        composeTestRule.onNodeWithTag(nrOfRowsValueTag).assertDoesNotExist()
        composeTestRule.onNodeWithTag(gaugeLengthValueTag).assertDoesNotExist()
    }

    @Test
    fun repositoryReturnsErrorDuringLoad_showsError() {
        repo.returnError = true
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag("LoadStateError").assertExists()
    }

    @Test
    fun repositoryReturnsNullDuringLoad_showsError() {
        repo.returnNull = true
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag("LoadStateNotFound").assertExists()
    }

    @Test
    fun repositoryReturnsErrorDuringDelete_showsErrorDialog() {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(deleteButtonTag).performClick()
        repo.returnError = true
        composeTestRule.onNodeWithTag(confirmButtonTag).performClick()

        composeTestRule.onNodeWithTag(deleteErrorDialogTag).assertExists()
    }

    @Test
    fun deleteErrorDialog_cancelClosesDialog() {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(deleteButtonTag).performClick()
        repo.returnError = true
        composeTestRule.onNodeWithTag(confirmButtonTag).performClick()

        composeTestRule.onNodeWithTag(deleteErrorDialogTag).assertExists()

        composeTestRule.onNodeWithTag("CancelErrorDialogButton").performClick()

        composeTestRule.onNodeWithTag(deleteErrorDialogTag).assertDoesNotExist()
    }

    @Test
    fun deleteErrorDialog_retryButtonTriggersRetry() {
        var onNavigateBackCalled = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { onNavigateBackCalled = true },
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(deleteButtonTag).performClick()
        repo.returnError = true
        composeTestRule.onNodeWithTag(confirmButtonTag).performClick()

        composeTestRule.onNodeWithTag(deleteErrorDialogTag).assertExists()
        repo.returnError = false
        composeTestRule.onNodeWithTag("RetryErrorDialogButton").performClick()

        assertTrue(onNavigateBackCalled)
    }

    @Test
    fun deleteSwatch_callsOnNavigateBack() {
        var onNavigateBackCalled = false
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = { onNavigateBackCalled = true },
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(optionsMenuTag).performClick()
        composeTestRule.onNodeWithTag(deleteButtonTag).performClick()
        composeTestRule.onNodeWithTag(confirmButtonTag).performClick()

        assertTrue(onNavigateBackCalled)
    }

    @Test
    fun editGaugeButtonClicked_showsDialog() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(editGaugeDialogTag).assertIsDisplayed()
    }

    @Test
    fun gaugeDialog_gaugeNullWithNoInput_saveButtonDisabled() = runTest {
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(saveGaugeButtonTag).assertIsNotEnabled()
    }

    @Test
    fun gaugeDialog_gaugeWithValidInput_saveButtonEnabled() = runTest {
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag(nrOfStitchesInputTag).performTextInput("13")
        composeTestRule.onNodeWithTag(nrOfRowsInputTag).performTextInput("21")
        composeTestRule.onNodeWithTag(gaugeLengthInputTag).performTextInput("10")

        composeTestRule.onNodeWithTag(saveGaugeButtonTag).assertIsEnabled()
    }

    @Test
    fun gaugeDialog_repositoryReturnsErrorDuringSave_showsError() {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        val stsNode = composeTestRule.onNodeWithTag(nrOfStitchesInputTag)
        stsNode.performTextReplacement("13")
        repo.returnError = true
        composeTestRule.onNodeWithTag(saveGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag("GaugeSaveError").assertExists()
    }

    @Test
    fun gaugeDialog_cancelButtonClicked_closesDialog() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag(cancelEditGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(editGaugeDialogTag).assertDoesNotExist()
    }

    @Test
    fun gaugeDialog_cancelButtonClicked_resetsState() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag(nrOfStitchesInputTag).performTextReplacement("13")
        composeTestRule.onNodeWithTag(nrOfRowsInputTag).performTextReplacement("21")
        composeTestRule.onNodeWithTag(gaugeLengthInputTag).performTextReplacement("10")

        composeTestRule.onNodeWithTag(cancelEditGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()

        val gauge = repo.swatchToReturn()!!.gauge!!
        composeTestRule.onNodeWithTag(nrOfStitchesInputTag)
            .assert(hasText(gauge.nrOfStitches.toString()))
        composeTestRule.onNodeWithTag(nrOfRowsInputTag).assert(hasText(gauge.nrOfRows.toString()))
        composeTestRule.onNodeWithTag(gaugeLengthInputTag)
            .assert(hasText(gauge.size.toString()))
    }

    @Test
    fun gaugeDialog_saveReturnsError_errorIsShownToUser() {
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(nrOfStitchesInputTag).performTextReplacement("13")
        composeTestRule.onNodeWithTag(nrOfRowsInputTag).performTextReplacement("21")
        composeTestRule.onNodeWithTag(gaugeLengthInputTag).performTextReplacement("10")

        repo.returnErrorDuringUpdate = true
        composeTestRule.onNodeWithTag(saveGaugeButtonTag).performClick()
        composeTestRule.onNodeWithTag(editGaugeDialogTag).assertIsDisplayed()

        composeTestRule.onNodeWithTag("GaugeSaveError").assertExists()
    }

    @Test
    fun gaugeDialog_saveReturnsError_gaugeIsNotUpdatedInDetailsView() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag(editGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(nrOfStitchesInputTag).performTextReplacement("50")
        composeTestRule.onNodeWithTag(nrOfRowsInputTag).performTextReplacement("37")
        composeTestRule.onNodeWithTag(gaugeLengthInputTag).performTextReplacement("16")

        repo.returnErrorDuringUpdate = true
        composeTestRule.onNodeWithTag(saveGaugeButtonTag).performClick()

        composeTestRule.onNodeWithTag(cancelEditGaugeButtonTag).performScrollTo().performClick()

        val gauge = repo.swatchToReturn()!!.gauge!!
        composeTestRule.onNodeWithTag(nrOfStitchesValueTag)
            .assert(hasText(gauge.nrOfStitches.toString()))
        composeTestRule.onNodeWithTag(nrOfRowsValueTag).assert(hasText(gauge.nrOfRows.toString()))
        composeTestRule.onNodeWithTag(gaugeLengthValueTag)
            .assert(hasText(gauge.size.toString(), substring = true))
    }
}

private fun defaultSwatchWithoutGauge() = Swatch.create(
    needleSize = KnittingNeedleSize.SIZE_10_0,
    pattern = null,
    yarn = null,
    notes = null,
    createdAt = EpochMillis(0),
    id = SwatchId(1),
    name = null
)

private const val editGaugeButtonTag = "EditGaugeButton"
private const val nrOfStitchesInputTag = "NrOfStitchesInput"
private const val nrOfRowsInputTag = "NrOfRowsInput"
private const val gaugeLengthInputTag = "GaugeLengthInput"
private const val saveGaugeButtonTag = "SaveGaugeButton"
private const val editGaugeDialogTag = "EditGaugeDialog"
private const val nrOfStitchesValueTag = "NrOfStitches"
private const val nrOfRowsValueTag = "NrOfRows"
private const val gaugeLengthValueTag = "GaugeLength"
private const val dismissButtonTag = "DismissButton"
private const val editButtonTag = "EditButton"
private const val optionsMenuTag = "OptionsMenu"
private const val viewMeasurementsDropDownItemTag = "ViewMeasurementsDropdownItem"
private const val nameValueTag = "Name"
private const val deleteButtonTag = "DeleteButton"
private const val confirmButtonTag = "ConfirmButton"
private const val deleteErrorDialogTag = "DeleteErrorDialog"
private const val cancelEditGaugeButtonTag = "CancelEditGaugeButton"
