@file:OptIn(ExperimentalCoroutinesApi::class)

package net.damschen.swatchit.test.ui.screens

import android.content.Context
import androidx.activity.ComponentActivity
import androidx.compose.ui.test.assertIsDisplayed
import androidx.compose.ui.test.assertIsEnabled
import androidx.compose.ui.test.assertIsNotDisplayed
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.isDisplayed
import androidx.compose.ui.test.junit4.createAndroidComposeRule
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.core.net.toUri
import androidx.test.core.app.ApplicationProvider
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.test.StandardTestDispatcher
import kotlinx.coroutines.test.advanceUntilIdle
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.infrastructure.services.AndroidPhotoStorageService
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.shared.testhelpers.FakeBitmapDecoder
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.shared.testhelpers.createTestImageFileInFileSystem
import net.damschen.swatchit.shared.testhelpers.MainDispatcherRule
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.screens.SwatchDetailsScreen
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.Assert.assertFalse
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog
import java.io.File

@RunWith(RobolectricTestRunner::class)
class SwatchDetailsScreenFileSystemTests {
    @get:Rule
    val composeTestRule = createAndroidComposeRule<ComponentActivity>()

    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    private lateinit var viewModel: EditSwatchViewModel
    private var itemId: Int = 1
    private lateinit var defaultSwatch: Swatch
    private var repo = FakeRepo()

    private lateinit var context: Context
    private lateinit var photoStorageService: PhotoStorageService

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initViewModel() {
        context = ApplicationProvider.getApplicationContext()
        val testDispatcher = StandardTestDispatcher(mainDispatcherRule.testDispatcher.scheduler)
        photoStorageService = AndroidPhotoStorageService(
            context, FakeBitmapDecoder(),
            testDispatcher
        )
        repo = FakeRepo()
        defaultSwatch = repo.swatchToReturn()!!
        itemId = repo.defaultId
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            photoStorageService,
            FakeUUIDProvider()
        )
    }

    @Test
    fun photo_swatchWithPhoto_showsPhotoCard() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchPhotoCaption").assertExists()
    }

    @Test
    fun photo_swatchWithoutPhoto_showsPickPhotoButton() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("PhotoPickerButton").assertExists()
    }

    @Test
    fun photo_swatchWithoutPhoto_deletePhotoButtonIsNotEnabled() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("DeletePhotoButton").assertIsNotEnabled()
    }

    @Test
    fun photo_swatchWithPhoto_deletePhotoButtonIsEnabled() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()

        composeTestRule.onNodeWithTag("DeletePhotoButton").assertIsEnabled()
    }

    @Test
    fun photo_swatchWithPhoto_showsPhoto() = runTest {
        createTestImageFileInFileSystem(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchBitmap").assertExists()
    }

    @Test
    fun photo_swatchWithoutPhoto_doesNotShowPhoto() = runTest {
        createTestImageFileInFileSystem(context)
        repo.swatchToReturn = { defaultSwatchWithoutGauge() }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchBitmap").performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun photo_swatchWithPhotoNotFoundInFileSystem_doesNotShowPhoto() = runTest {
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("SwatchBitmap").performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun photo_swatchWithPhotoNotFoundInFileSystem_showsErrorMessage() = runTest {
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("BitmapNotFoundError").assertExists()
    }

    @Test
    fun deletePhoto_swatchWithPhoto_removesPhoto() = runTest {
        createTestImageFileInFileSystem(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("DeletePhotoButton").performScrollTo().performClick()
        composeTestRule.waitUntil {
            composeTestRule.onNodeWithTag("ConfirmDeleteDialog").isDisplayed()
        }
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        composeTestRule.onNodeWithTag("SwatchBitmap").performScrollTo().assertIsNotDisplayed()
    }

    @Test
    fun deletePhoto_swatchWithPhotoCancelDeletion_doesNotRemovePhoto() = runTest {
        createTestImageFileInFileSystem(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        composeTestRule.onNodeWithTag("DeletePhotoButton").performScrollTo().performClick()
        composeTestRule.waitUntil {
            composeTestRule.onNodeWithTag("ConfirmDeleteDialog").isDisplayed()
        }
        composeTestRule.onNodeWithTag("CancelDeleteButton").performClick()

        composeTestRule.onNodeWithTag("SwatchBitmap").assertExists()
    }

    @Test
    fun photoPicker_errorDuringUpdate_showsErrorMessage() = runTest {
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        repo.returnErrorDuringUpdate = true
        val testUri = "content://media/external/images/media/1".toUri()
        viewModel.updateSwatchPhoto(testUri)

        advanceUntilIdle()
        composeTestRule.onNodeWithTag("PhotoUpdateError").performScrollTo().assertIsDisplayed()
    }

    @Test
    fun deletePhoto_errorDuringUpdate_showsErrorMessage() = runTest {
        createTestImageFileInFileSystem(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }
        repo.returnErrorDuringUpdate = true
        composeTestRule.onNodeWithTag("DeletePhotoButton").performScrollTo().performClick()
        composeTestRule.waitUntil {
            composeTestRule.onNodeWithTag("ConfirmDeleteDialog").isDisplayed()
        }
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()
        advanceUntilIdle()

        composeTestRule.onNodeWithTag("PhotoUpdateError").performScrollTo().assertIsDisplayed()
        composeTestRule.onNodeWithTag("BitmapNotFoundError").assertExists()
    }

    @Test
    fun deleteSwatch_validInput_removesSwatchFromDb() = runTest {
        createTestImageFileInFileSystem(context)
        repo.swatchToReturn =
            { defaultSwatchWithoutGauge().withUpdatedPhoto(Photo(FakeUUIDProvider.defaultUUID)) }
        viewModel.loadSwatch()
        advanceUntilIdle()
        composeTestRule.setContent {
            SwatchDetailsScreen(
                viewModel,
                onNavigateBack = {},
                onEdit = {},
                onViewMeasurements = {},
                onViewCalculations = {})
        }

        composeTestRule.onNodeWithTag("OptionsMenu").performClick()
        composeTestRule.onNodeWithTag("DeleteButton").performClick()
        composeTestRule.onNodeWithTag("ConfirmButton").performClick()

        advanceUntilIdle()

        val savedFile = File(context.filesDir, "images/${FakeUUIDProvider.defaultUUID}.jpg")
        assertFalse(savedFile.exists())
    }
}

private fun defaultSwatchWithoutGauge() = Swatch.create(
    needleSize = KnittingNeedleSize.SIZE_10_0,
    pattern = null,
    yarn = null,
    notes = null,
    createdAt = EpochMillis(0),
    id = SwatchId(1),
    name = null
)