package net.damschen.swatchit.test.ui.screens

import androidx.compose.ui.semantics.SemanticsProperties
import androidx.compose.ui.test.SemanticsMatcher
import androidx.compose.ui.test.assert
import androidx.compose.ui.test.assertTextContains
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onChild
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.compose.ui.test.performTextInput
import androidx.compose.ui.text.AnnotatedString
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.test.testHelpers.FakePhotoStorageService
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.screens.GaugeCalculationsScreen
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog

@RunWith(RobolectricTestRunner::class)
class GaugeCalculationsScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    private lateinit var viewModel: EditSwatchViewModel
    private var itemId: Int = 1
    private lateinit var defaultSwatch: Swatch
    private var repo = FakeRepo()

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initViewModel() {
        repo = FakeRepo()
        defaultSwatch = repo.swatchToReturn()!!
        itemId = repo.defaultId
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
    }

    @Test
    fun backButtonClicked_callsOnDismiss() {
        var dismissed = false
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = { dismissed = true })
        }

        composeTestRule.onNodeWithTag("BackButton").performClick()

        assertTrue(dismissed)
    }

    @Test
    fun repositoryReturnsErrorDuringLoad_showsError() {
        repo.returnError = true
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("LoadStateError").assertExists()
    }

    @Test
    fun repositoryReturnsNullDuringLoad_showsError() {
        repo.returnNull = true
        viewModel = EditSwatchViewModel(
            itemId,
            repo,
            FakeDateTimeProvider(),
            FakePhotoStorageService(),
            FakeUUIDProvider()
        )
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("LoadStateNotFound").assertExists()
    }

    @Test
    fun calculationsColumn_validWidthInput_printsStitches() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CalculationWidth").performScrollTo().performTextInput("100")
        composeTestRule.onNodeWithTag("CalculationStitches").assertTextContains("200")
    }

    @Test
    fun calculationsColumn_validStitchesInput_printsWidth() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CalculationStitches").performScrollTo()
            .performTextInput("200")
        composeTestRule.onNodeWithTag("CalculationWidth").assertTextContains("100")
    }

    @Test
    fun calculationsColumn_bigWidthValueCausesLongOverflow_showsErrorMessage() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CalculationWidth").performScrollTo()
            .performTextInput(Long.MAX_VALUE.toDouble().toPlainStringWithoutTrailingZeros())
        composeTestRule.onNodeWithTag("CalculationStitches")
            .assertTextContains("Could not calculate", substring = true)
    }

    @Test
    fun calculationsColumn_validHeightInput_printsRows() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CalculationHeight").performScrollTo().performTextInput("100")
        composeTestRule.onNodeWithTag("CalculationRows").assertTextContains("300")
    }

    @Test
    fun calculationsColumn_validRowInput_printsHeight() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CalculationRows").performScrollTo()
            .performTextInput("300")
        composeTestRule.onNodeWithTag("CalculationHeight").assertTextContains("100")
    }

    @Test
    fun calculationsColumn_bigHeightValueCausesLongOverflow_showsErrorMessage() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CalculationHeight").performScrollTo()
            .performTextInput(Long.MAX_VALUE.toDouble().toPlainStringWithoutTrailingZeros())
        composeTestRule.onNodeWithTag("CalculationRows")
            .assertTextContains("Could not calculate", substring = true)
    }

    @Test
    fun calculationsColumn_clearWidthButtonClicked_clearsInput() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        val node = composeTestRule.onNodeWithTag("CalculationWidth")
        node.performScrollTo().performTextInput("100")
        node.onChild().performClick()
        node.assert(
            SemanticsMatcher.expectValue(
                SemanticsProperties.EditableText,
                AnnotatedString("")
            )
        )
    }

    @Test
    fun calculationsColumn_clearStitchesButtonClicked_clearsInput() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        val node = composeTestRule.onNodeWithTag("CalculationStitches")
        node.performScrollTo().performTextInput("100")
        node.onChild().performClick()
        node.assert(
            SemanticsMatcher.expectValue(
                SemanticsProperties.EditableText,
                AnnotatedString("")
            )
        )
    }

    @Test
    fun calculationsColumn_clearRowsButtonClicked_clearsInput() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        val node = composeTestRule.onNodeWithTag("CalculationRows")
        node.performScrollTo().performTextInput("100")
        node.onChild().performClick()
        node.assert(
            SemanticsMatcher.expectValue(
                SemanticsProperties.EditableText,
                AnnotatedString("")
            )
        )
    }

    @Test
    fun calculationsColumn_clearHeightButtonClicked_clearsInput() = runTest {
        composeTestRule.setContent {
            GaugeCalculationsScreen(
                viewModel,
                onNavigateBack = {})
        }

        val node = composeTestRule.onNodeWithTag("CalculationHeight")
        node.performScrollTo().performTextInput("100")
        node.onChild().performClick()
        node.assert(
            SemanticsMatcher.expectValue(
                SemanticsProperties.EditableText,
                AnnotatedString("")
            )
        )
    }
}