package net.damschen.swatchit.test.ui.screens

import androidx.compose.ui.test.assertIsEnabled
import androidx.compose.ui.test.assertIsNotEnabled
import androidx.compose.ui.test.hasText
import androidx.compose.ui.test.isDisplayed
import androidx.compose.ui.test.junit4.createComposeRule
import androidx.compose.ui.test.onNodeWithContentDescription
import androidx.compose.ui.test.onNodeWithTag
import androidx.compose.ui.test.performClick
import androidx.compose.ui.test.performScrollTo
import androidx.compose.ui.test.performTextClearance
import androidx.compose.ui.test.performTextInput
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.test.testHelpers.database.FakeRepo
import net.damschen.swatchit.ui.screens.AddSwatchScreen
import net.damschen.swatchit.ui.viewmodels.AddSwatchViewModel
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.shadows.ShadowLog

@RunWith(RobolectricTestRunner::class)
class AddSwatchScreenTests {
    @get:Rule
    val composeTestRule = createComposeRule()

    private var repo = FakeRepo()
    private lateinit var viewModel: AddSwatchViewModel

    @Before
    @Throws(Exception::class)
    fun setUp() {
        ShadowLog.stream = System.out
    }

    @Before
    fun initViewModel() {
        viewModel = AddSwatchViewModel(repo, FakeDateTimeProvider())
    }

    @Test
    fun backButtonClicked_callsOnNavigateBack() {
        var cancelled = false
        composeTestRule.setContent {
            AddSwatchScreen(viewModel, onNavigateBack = { cancelled = true })
        }

        composeTestRule.onNodeWithTag(cancelButtonTag).performClick()

        assertTrue(cancelled)
    }

    @Test
    fun noFieldsSetManually_saveButtonIsEnabled() {
        composeTestRule.setContent {
            AddSwatchScreen(viewModel, onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag(addButtonTag).assertIsEnabled()
    }

    @Test
    fun createdAtValid_datePickerOkButtonIsEnabled() {
        composeTestRule.setContent {
            AddSwatchScreen(viewModel, onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CreatedAt").performScrollTo().performClick()
        // Wait for the date picker dialog to appear
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("DatePickerDialog").isDisplayed()
        }
        composeTestRule.onNode(hasText("Selected date")).performClick()
        composeTestRule.onNodeWithContentDescription("Switch to text input mode").performClick()
        composeTestRule.onNodeWithContentDescription("Date, MM/DD/YYYY")
            .performTextInput("12/12/2012")
        composeTestRule.onNodeWithTag("DatePickerOkButton").assertIsNotEnabled()
    }

    @Test
    fun createdAtNotSet_datePickerOkButtonIsDisabled() {
        composeTestRule.setContent {
            AddSwatchScreen(viewModel, onNavigateBack = {})
        }

        composeTestRule.onNodeWithTag("CreatedAt").performScrollTo().performClick()
        // Wait for the date picker dialog to appear
        composeTestRule.waitUntil(timeoutMillis = 5000) {
            composeTestRule.onNodeWithTag("DatePickerDialog").isDisplayed()
        }
        composeTestRule.onNode(hasText("Selected date")).performClick()
        composeTestRule.onNodeWithContentDescription("Switch to text input mode").performClick()
        composeTestRule.onNodeWithContentDescription("Date, MM/DD/YYYY").performTextClearance()
        composeTestRule.onNodeWithTag("DatePickerOkButton").assertIsNotEnabled()
    }

    @Test
    fun repositoryReturnsErrorDuringSave_showsErrorDialog() {
        composeTestRule.setContent {
            AddSwatchScreen(
                viewModel,
                onNavigateBack = {},
            )
        }

        repo.returnError = true
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag(addErrorDialogTag).assertExists()
    }

    @Test
    fun addErrorDialog_cancelClosesDialog() {
        composeTestRule.setContent {
            AddSwatchScreen(
                viewModel,
                onNavigateBack = {},
            )
        }

        repo.returnError = true
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag(addErrorDialogTag).assertExists()

        composeTestRule.onNodeWithTag("CancelErrorDialogButton").performClick()

        composeTestRule.onNodeWithTag(addErrorDialogTag).assertDoesNotExist()
    }

    @Test
    fun addErrorDialog_retryButtonTriggersRetry() {
        var onNavigateBackCalled = false
        composeTestRule.setContent {
            AddSwatchScreen(
                viewModel,
                onNavigateBack = { onNavigateBackCalled = true },
            )
        }

        repo.returnError = true
        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        composeTestRule.onNodeWithTag(addErrorDialogTag).assertExists()
        repo.returnError = false
        composeTestRule.onNodeWithTag("RetryErrorDialogButton").performClick()

        assertTrue(onNavigateBackCalled)
    }

    @Test
    fun addSwatch_callsOnNavigateBack() {
        var onNavigateBackCalled = false
        composeTestRule.setContent {
            AddSwatchScreen(
                viewModel,
                onNavigateBack = { onNavigateBackCalled = true },
            )
        }

        composeTestRule.onNodeWithTag(addButtonTag).performClick()

        assertTrue(onNavigateBackCalled)
    }

    private val cancelButtonTag = "CancelButton"
    private val addButtonTag = "AddSwatchButton"
    private val addErrorDialogTag = "AddErrorDialog"
}