package net.damschen.swatchit.test.ui.models

import junit.framework.TestCase.assertEquals
import junit.framework.TestCase.assertNull
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.SwatchName
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.aggregates.swatch.YarnManufacturer
import net.damschen.swatchit.domain.aggregates.swatch.YarnName
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.ui.models.SwatchState
import org.junit.Test
import java.time.LocalDate
import java.time.ZoneId

class SwatchStateTest {
    @Test
    fun from_validSwatch_createsSwatchState() {
        val swatch = Swatch.create(
            needleSize = defaultNeedleSize,
            pattern = defaultPattern,
            yarn = defaultYarn,
            notes = defaultNotes,
            name = defaultName,
            createdAt = defaultCreatedAt,
            id = defaultId
        ).withNewMeasurements(defaultMeasurements).withUpdatedGauge(defaultGauge)
            .withUpdatedPhoto(defaultPhoto)

        val state = SwatchState.from(swatch)

        assertEquals(defaultNeedleSize, state?.needleSize)
        assertEquals(defaultPattern, state?.pattern)
        assertEquals(defaultGauge, state?.gauge)
        assertEquals(defaultYarn, state?.yarn)
        assertEquals(defaultNotes, state?.notes)
        assertEquals(defaultMeasurements, state?.measurements)
        assertEquals(defaultName, state?.name)
        assertEquals(defaultCreatedAt, state?.createdAt)
        assertEquals(defaultId, state?.id)
        assertEquals(defaultPhoto, state?.photo)
    }

    @Test
    fun from_nullSwatch_returnsNull() {
        val state = SwatchState.from(null)

        assertNull(state)
    }


    @Test
    fun toSwatch_withAllFields_createsSwatch() {
        val state = SwatchState(
            needleSize = defaultNeedleSize,
            pattern = defaultPattern,
            gauge = defaultGauge,
            yarn = defaultYarn,
            notes = defaultNotes,
            measurements = defaultMeasurements,
            name = defaultName,
            createdAt = defaultCreatedAt,
            id = defaultId,
            photo = defaultPhoto
        )

        val swatch = state.toSwatch()

        assertEquals(defaultNeedleSize, swatch.needleSize)
        assertEquals(defaultPattern, swatch.pattern)
        assertEquals(defaultYarn, swatch.yarn)
        assertEquals(defaultNotes, swatch.notes)
        assertEquals(defaultMeasurements, swatch.measurements)
        assertEquals(defaultName, swatch.name)
        assertEquals(defaultCreatedAt, swatch.createdAt)
        assertEquals(defaultId, swatch.id)
        assertEquals(defaultPhoto, swatch.photo)
    }

    @Test
    fun from_toSwatch_roundTrip_preservesData() {

        val originalSwatch = Swatch.create(
            needleSize = defaultNeedleSize,
            pattern = defaultPattern,
            yarn = defaultYarn,
            notes = defaultNotes,
            name = defaultName,
            createdAt = defaultCreatedAt,
            id = defaultId
        ).withNewMeasurements(defaultMeasurements).withUpdatedGauge(defaultGauge)
            .withUpdatedPhoto(defaultPhoto)

        val state = SwatchState.from(originalSwatch)
        val resultSwatch = state?.toSwatch()

        assertEquals(originalSwatch.needleSize, resultSwatch?.needleSize)
        assertEquals(originalSwatch.pattern, resultSwatch?.pattern)
        assertEquals(originalSwatch.yarn, resultSwatch?.yarn)
        assertEquals(originalSwatch.notes, resultSwatch?.notes)
        assertEquals(originalSwatch.measurements, resultSwatch?.measurements)
        assertEquals(originalSwatch.name, resultSwatch?.name)
        assertEquals(originalSwatch.createdAt, resultSwatch?.createdAt)
        assertEquals(originalSwatch.id, resultSwatch?.id)
        assertEquals(originalSwatch.photo, resultSwatch?.photo)
    }
}

private val defaultNeedleSize = KnittingNeedleSize.SIZE_0_75
private val defaultPattern = Pattern.create("testPattern")
private val defaultId = SwatchId(1)
private val defaultPhoto = Photo(FakeUUIDProvider.defaultUUID)
private val defaultGauge =
    Gauge(GaugeCount(20), GaugeCount(30), GaugeSize(10.0))
private val defaultYarn =
    Yarn.create(YarnName.create("Yarn Name"), YarnManufacturer.create("Yarn Manufacturer"))
private val defaultNotes = Notes.create("Test notes!")
private val defaultCreatedAt = EpochMillis(
    LocalDate.of(2025, 2, 17).atStartOfDay(
        ZoneId.of("UTC")
    ).toInstant().toEpochMilli()
)
private val defaultName = SwatchName.create("Fake Name")

private val defaultMeasurement =
    Measurement.Rows(
        GaugeCount(34),
        GaugeSize(13.0)
    )
private val defaultMeasurements = listOf(defaultMeasurement)