package net.damschen.swatchit.test.ui.models

import junit.framework.TestCase.assertEquals
import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.SwatchName
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.aggregates.swatch.YarnManufacturer
import net.damschen.swatchit.domain.aggregates.swatch.YarnName
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.ui.enums.toUiEnum
import net.damschen.swatchit.ui.models.GaugeState
import net.damschen.swatchit.ui.models.SwatchListItem
import org.junit.Assert.assertThrows
import org.junit.Test
import java.time.LocalDate
import java.time.ZoneId

class SwatchListItemTests {

    @Test
    fun fromSwatch_createsItemFromSwatch() {
        val swatch = Swatch.create(
            needleSize = defaultNeedleSize,
            pattern = defaultPattern,
            yarn = defaultYarn,
            notes = defaultNotes,
            name = defaultName,
            createdAt = defaultCreatedAt,
            id = defaultId
        ).withNewMeasurements(defaultMeasurements).withUpdatedGauge(defaultGauge)
            .withUpdatedPhoto(defaultPhoto)
        val item = SwatchListItem.fromSwatch(swatch)

        assertEquals(defaultNeedleSize.toUiEnum(), item.needleSize)
        assertEquals(defaultId.value, item.id)
        assertEquals(defaultName?.value, item.name)
        assertEquals(defaultYarn?.name?.value, item.yarnName)
        assertEquals(defaultYarn?.manufacturer?.value, item.yarnManufacturer)
        assertEquals(GaugeState.fromGauge(defaultGauge), item.gaugeState)
        assertEquals(defaultPhoto.fileName, item.photoFileName)
    }

    @Test
    fun fromSwatch_idNull_throws() {
        val swatch = Swatch.create(
            needleSize = defaultNeedleSize,
            createdAt = defaultCreatedAt
        )
        assertThrows(IllegalArgumentException::class.java) { SwatchListItem.fromSwatch(swatch) }
    }
}

private val defaultNeedleSize = KnittingNeedleSize.SIZE_0_75
private val defaultPattern = Pattern.create("testPattern")
private val defaultId = SwatchId(1)
private val defaultPhoto = Photo(FakeUUIDProvider.defaultUUID)
private val defaultGauge =
    Gauge(GaugeCount(20), GaugeCount(30), GaugeSize(10.0))
private val defaultYarn =
    Yarn.create(YarnName.create("Yarn Name"), YarnManufacturer.create("Yarn Manufacturer"))
private val defaultNotes = Notes.create("Test notes!")
private val defaultCreatedAt = EpochMillis(
    LocalDate.of(2025, 2, 17).atStartOfDay(
        ZoneId.of("UTC")
    ).toInstant().toEpochMilli()
)
private val defaultName = SwatchName.create("Fake Name")

private val defaultMeasurement =
    Measurement.Rows(
        GaugeCount(34),
        GaugeSize(13.0)
    )
private val defaultMeasurements = listOf(defaultMeasurement)