package net.damschen.swatchit.test.ui.models

import junit.framework.TestCase.assertEquals
import junit.framework.TestCase.assertFalse
import junit.framework.TestCase.assertTrue
import net.damschen.swatchit.shared.testhelpers.FakeDateTimeProvider
import net.damschen.swatchit.shared.testhelpers.MainDispatcherRule
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import net.damschen.swatchit.ui.models.SwatchFormStateManager
import net.damschen.swatchit.ui.models.ValidatedInput
import org.junit.Rule
import org.junit.Test

class BaseSwatchViewModelTests {
    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    @Test
    fun init_setsDefaultState() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())

        val state = sut.swatchFormState.value
        assertEquals("", state.name.value)
        assertEquals("", state.yarnManufacturer.value)
        assertEquals("", state.yarnName.value)
        assertEquals("", state.pattern.value)
        assertEquals("", state.notes.value)
        assertEquals(FakeDateTimeProvider().now().time, state.createdAt)
    }

    @Test
    fun resetState_resetsToDefaultState() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())

        sut.onNameChange("Test Name")
        sut.onYarnManufacturerChange("Test Manufacturer")
        sut.onNotesChange("Test Notes")

        sut.resetFormState()

        val state = sut.swatchFormState.value
        assertEquals("", state.name.value)
        assertEquals("", state.yarnManufacturer.value)
        assertEquals("", state.yarnName.value)
        assertEquals("", state.pattern.value)
        assertEquals("", state.notes.value)
    }

    @Test
    fun onNameChange_validInput_updatesNameWithNoError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val validName = "Valid Swatch Name"

        sut.onNameChange(validName)

        val nameState = sut.swatchFormState.value.name
        assertEquals(validName, nameState.value)
        assertTrue(nameState is ValidatedInput.Valid)
    }

    @Test
    fun onNameChange_invalidInput_updatesNameWithError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val invalidName = "A".repeat(51)

        sut.onNameChange(invalidName)

        val nameState = sut.swatchFormState.value.name
        assertEquals(invalidName, nameState.value)
        assertFalse(nameState is ValidatedInput.Valid)
    }

    @Test
    fun onYarnManufacturerChange_validInput_updatesYarnManufacturerWithNoError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val validManufacturer = "Valid Manufacturer"

        sut.onYarnManufacturerChange(validManufacturer)

        val manufacturerState = sut.swatchFormState.value.yarnManufacturer
        assertEquals(validManufacturer, manufacturerState.value)
        assertTrue(manufacturerState is ValidatedInput.Valid)
    }

    @Test
    fun onYarnManufacturerChange_invalidInput_updatesYarnManufacturerWithError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val invalidManufacturer = "A".repeat(51)

        sut.onYarnManufacturerChange(invalidManufacturer)

        val manufacturerState = sut.swatchFormState.value.yarnManufacturer
        assertEquals(invalidManufacturer, manufacturerState.value)
        assertFalse(manufacturerState is ValidatedInput.Valid)
    }

    @Test
    fun onYarnNameChange_validInput_updatesYarnNameWithNoError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val validYarnName = "Valid Yarn Name"

        sut.onYarnNameChange(validYarnName)

        val yarnNameState = sut.swatchFormState.value.yarnName
        assertEquals(validYarnName, yarnNameState.value)
        assertTrue(yarnNameState is ValidatedInput.Valid)
    }

    @Test
    fun onYarnNameChange_invalidInput_updatesYarnNameWithError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val invalidYarnName = "A".repeat(51)

        sut.onYarnNameChange(invalidYarnName)

        val yarnNameState = sut.swatchFormState.value.yarnName
        assertEquals(invalidYarnName, yarnNameState.value)
        assertFalse(yarnNameState is ValidatedInput.Valid)
    }

    @Test
    fun onNeedleSizeChange_updatesNeedleSize() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val needleSize = KnittingNeedleSize.SIZE_10_0

        sut.onNeedleSizeChange(needleSize)

        assertEquals(needleSize, sut.swatchFormState.value.needleSize)
    }

    @Test
    fun onPatternChange_validInput_updatesPatternWithNoError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val validPattern = "Valid Pattern"

        sut.onPatternChange(validPattern)

        val patternState = sut.swatchFormState.value.pattern
        assertEquals(validPattern, patternState.value)
        assertTrue(patternState is ValidatedInput.Valid)
    }

    @Test
    fun onPatternChange_invalidInput_updatesPatternWithError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val invalidPattern = "A".repeat(151)

        sut.onPatternChange(invalidPattern)

        val patternState = sut.swatchFormState.value.pattern
        assertEquals(invalidPattern, patternState.value)
        assertFalse(patternState is ValidatedInput.Valid)
    }

    @Test
    fun onNotesChange_validInput_updatesNotesWithNoError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val validNotes = "These are valid notes"

        sut.onNotesChange(validNotes)

        val notesState = sut.swatchFormState.value.notes
        assertEquals(validNotes, notesState.value)
        assertTrue(notesState is ValidatedInput.Valid)
    }

    @Test
    fun onNotesChange_invalidInput_updatesNotesWithError() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val invalidNotes = "x".repeat(101)

        sut.onNotesChange(invalidNotes)

        val notesState = sut.swatchFormState.value.notes
        assertEquals(invalidNotes, notesState.value)
    }

    @Test
    fun onCreatedAtChange_updatesCreatedAt() {
        val sut = SwatchFormStateManager(FakeDateTimeProvider())
        val newTimestamp = 1234567890L

        sut.onCreatedAtChange(newTimestamp)

        assertEquals(newTimestamp, sut.swatchFormState.value.createdAt)
    }
}