package net.damschen.swatchit.integrationTest.infrastrcuture.services

import android.content.Context
import android.database.sqlite.SQLiteDatabase
import android.net.Uri
import androidx.core.net.toFile
import dagger.hilt.android.qualifiers.ApplicationContext
import dagger.hilt.android.testing.HiltAndroidRule
import dagger.hilt.android.testing.HiltAndroidTest
import dagger.hilt.android.testing.HiltTestApplication
import kotlinx.coroutines.test.StandardTestDispatcher
import kotlinx.coroutines.test.runTest
import net.damschen.swatchit.infrastructure.database.AppDatabase
import net.damschen.swatchit.infrastructure.database.KnittingNeedleSize
import net.damschen.swatchit.infrastructure.database.SwatchDao
import net.damschen.swatchit.infrastructure.database.SwatchEntity
import net.damschen.swatchit.infrastructure.resultWrappers.ExportResult
import net.damschen.swatchit.infrastructure.services.AndroidDataExportService
import net.damschen.swatchit.infrastructure.services.AndroidZipper
import net.damschen.swatchit.infrastructure.services.DataExportService
import net.damschen.swatchit.shared.testhelpers.FakeUUIDProvider
import net.damschen.swatchit.shared.testhelpers.FakeZipper
import net.damschen.swatchit.shared.testhelpers.createTestImageFileInFileSystem
import net.damschen.swatchit.shared.testhelpers.MainDispatcherRule
import org.junit.After
import org.junit.Assert.assertFalse
import org.junit.Assert.assertTrue
import org.junit.Before
import org.junit.Rule
import org.junit.Test
import org.junit.runner.RunWith
import org.robolectric.RobolectricTestRunner
import org.robolectric.annotation.Config
import java.io.File
import java.util.zip.ZipFile
import javax.inject.Inject
import javax.inject.Named

@RunWith(RobolectricTestRunner::class)
@HiltAndroidTest
@Config(application = HiltTestApplication::class)
class DataExportServiceTests {

    @get:Rule
    var hiltRule = HiltAndroidRule(this)

    @get:Rule
    val mainDispatcherRule = MainDispatcherRule()

    lateinit var dataExportService: DataExportService
    lateinit var backupDir: File

    @Inject
    @ApplicationContext
    lateinit var context: Context

    @Inject
    @Named("test_db")
    lateinit var database: AppDatabase

    @Inject
    lateinit var swatchDao: SwatchDao

    @Before
    fun setup() {
        hiltRule.inject()
        val testDispatcher = StandardTestDispatcher(mainDispatcherRule.testDispatcher.scheduler)
        dataExportService = AndroidDataExportService(
            context, database, AndroidZipper(),
            testDispatcher
        )
        backupDir = File(context.cacheDir, "bkp")
    }

    @After
    fun closeDb() {
        database.close()
    }

    @Test
    fun zipDatabaseAndPictures_savesDatabaseBackup() = runTest {
        swatchDao.insert(createSwatchEntity())
        val result = dataExportService.zipDatabaseAndPictures()

        assertTrue(result is ExportResult.Success)

        val zipUri = (result as ExportResult.Success).zipUri

        val extractionDir = File(context.cacheDir, "test_unzipped")
        if (extractionDir.exists()) extractionDir.deleteRecursively()
        else extractionDir.mkdirs()

        unzip(zipUri, extractionDir)

        val dbFile = File(extractionDir, "swatchit_database_backup.db")
        assertTrue(dbFile.exists())

        val unzippedDb =
            SQLiteDatabase.openDatabase(dbFile.path, null, SQLiteDatabase.OPEN_READONLY)
        val cursor = unzippedDb.rawQuery("SELECT COUNT(*) FROM swatches", null)
        cursor.moveToFirst()
        val count = cursor.getInt(0)
        cursor.close()
        unzippedDb.close()

        assertTrue(count > 0)

        extractionDir.deleteRecursively()
    }

    @Test
    fun zipDatabaseAndPictures_copiesImagesFolder() = runTest {
        createTestImageFileInFileSystem(context)
        val result = dataExportService.zipDatabaseAndPictures()

        val zipUri = (result as ExportResult.Success).zipUri

        val extractionDir = File(context.cacheDir, "test_unzipped")
        if (extractionDir.exists()) extractionDir.deleteRecursively()
        else extractionDir.mkdirs()

        unzip(zipUri, extractionDir)

        val savedFolder = File(extractionDir, "images_backup")
        assertTrue(savedFolder.exists())
        assertTrue(savedFolder.isDirectory)
        val savedImage = File(savedFolder.path, "${FakeUUIDProvider.defaultUUID}.jpg")
        assertTrue(savedImage.exists())
        assertTrue(savedImage.isFile)

        extractionDir.deleteRecursively()
    }

    @Test
    fun zipDatabaseAndPictures_zipsBackupFolder() = runTest {
        val result = dataExportService.zipDatabaseAndPictures()

        val backupFile = File(context.cacheDir, DataExportService.BACKUP_FILE_NAME)
        assertTrue(backupFile.exists())
        assertTrue(backupFile.isFile)
        assertTrue(backupFile.length() > 0)
        assertTrue(result is ExportResult.Success)
    }

    @Test
    fun zipDatabaseAndPictures_cleansUpBackupDir() = runTest {
        dataExportService.zipDatabaseAndPictures()

        assertFalse(backupDir.exists())
    }

    @Test
    fun zipDatabaseAndPictures_errorDuringZipping_returnsError() = runTest {
        val zipper = FakeZipper()
        zipper.shouldThrow = true
        dataExportService = AndroidDataExportService(
            context,
            database,
            zipper,
            StandardTestDispatcher(testScheduler)
        )
        val result = dataExportService.zipDatabaseAndPictures()

        assertTrue(result is ExportResult.Error)
    }
}

private fun unzip(zipUri: Uri, extractionDir: File) {
    ZipFile(zipUri.toFile()).use { zip ->
        zip.entries().asSequence().forEach { entry ->
            zip.getInputStream(entry).use { input ->
                val filePath = File(extractionDir, entry.name)
                filePath.parentFile?.mkdirs()
                filePath.outputStream().use { output ->
                    input.copyTo(output)
                }
            }
        }
    }
}

private fun createSwatchEntity(): SwatchEntity {
    return SwatchEntity(
        needleSize = KnittingNeedleSize.SIZE_2_5,
        yarnName = "",
        yarnManufacturer = "",
        nrOfStitches = null,
        nrOfRows = null,
        gaugeLength = null,
        createdAt = 25L,
        name = "",
        pattern = "",
        notes = "",
        photoUUID = null
    )
}