package net.damschen.swatchit.ui.viewmodels

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.SwatchListItem
import javax.inject.Inject

@HiltViewModel
class SwatchListViewModel @Inject constructor(private val swatchRepository: SwatchRepository) :
    ViewModel() {
    private val _swatches = MutableStateFlow<List<SwatchListItem>>(emptyList())
    val swatches: StateFlow<List<SwatchListItem>> = _swatches

    private val _loadState = MutableStateFlow<LoadState>(LoadState.Initial)
    val loadState: StateFlow<LoadState> = _loadState.asStateFlow()

    init {
        loadSwatches()
    }

    internal fun loadSwatches() {
        viewModelScope.launch {
            _loadState.value = LoadState.Loading
            swatchRepository.getAll().collect { result ->
                when (result) {
                    is DatabaseResult.Success -> {
                        _swatches.value = result.data.map { swatch ->
                            SwatchListItem.fromSwatch(swatch)
                        }
                        _loadState.value = LoadState.Success
                    }

                    is DatabaseResult.Error -> _loadState.value =
                        LoadState.Error(result.exception.message)
                }
            }
        }
    }
}